/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import hirondelle.date4j.DateTime;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * This class represents a segment within a track. It encapsulates all data
 * (trackpoints, waypoints and heartrate data.)
 * @author UserXP
 */
public class TrackSegment
{
    private static int                  nextSegmentId=0;
    
    private int                         trackSegmentNo;
    private ArrayList<TrackLogPoint>    trackPoints;
    private ArrayList<TrackLogPoint>    wayPoints;

    /** Array with all heart rate values of all tracks in the log */
    private ArrayList<HeartRatePoint>   heartRatePoints;
    
   
    
    /**
     * Constructor. Creates the arrays.
     */
    public TrackSegment()
    {
        trackSegmentNo      =nextSegmentId;
        nextSegmentId++;
        
        trackPoints         =new ArrayList<TrackLogPoint>();
        wayPoints           =new ArrayList<TrackLogPoint>();
        heartRatePoints     =new ArrayList<HeartRatePoint>(); 
    }
    
    /**
     * This method resets the Segment housekeeping. Should be called 
     * for each track.
     */
    public static void reset()
    {
        nextSegmentId=0;
    }
    
    /**
     * This method adds a track point to the track log 
     * @param point The track log point to add
     */
    public void appendTrackpoint(TrackLogPoint point)
    {
        this.trackPoints.add(point);
    }

    /**
     * This method adds a point to the waypoint log
     * @param point Waypoint log point to add
     */
    public void appendWaypoint(TrackLogPoint point)
    {
        this.wayPoints.add(point);
    }
    
    /**
     * This method returns the array of heart rate values
     * @param points The array of heart rate values
     */
    public void appendHeartRatePoints(ArrayList<HeartRatePoint> points)
    {
        this.heartRatePoints.addAll(points);
    }

    /**
     * This method finishes the segment after appending all data.
     */
    public void finish()
    {
        this.parseHeartRates();
    }
    
    /**
     * This method parses the logs and adorns a heart rate value to the 
     * track log points.
     */
    private void parseHeartRates()
    {
        Iterator<TrackLogPoint>     trackLogIterator;
        Iterator<HeartRatePoint>    heartRateIterator;
        TrackLogPoint               trackPoint;
        HeartRatePoint              heartRatePoint;
        boolean                     exit;
        DateTime                    trackPointDateTime;
        DateTime                    heartRateDateTime;
        double                      sum;
        int                         count;
        
        // initial values
        heartRatePoint      =null;
        trackPoint          =null;
        
        // Obtain the iterators
        trackLogIterator    =this.trackPoints.iterator();
        heartRateIterator   =this.heartRatePoints.iterator();

        // point of starting
        exit=false;
        
        // Get the 1st heart rate
        if (heartRateIterator.hasNext())
        {
            heartRatePoint=heartRateIterator.next();
        }
        else
        {
            heartRatePoint=null;
            exit=true;
        }
        
        // get the 1st track point that is not a waypoint
        if (trackLogIterator.hasNext())
        {
            trackPoint=trackLogIterator.next();
        }
        else
        {
            trackPoint=null;
            exit=true;
        }
        
        // initialise sum and count for average calculation
        sum=0.0;
        count=0;
        
        while (!exit)
        {
            // get the timestamps
            trackPointDateTime  =trackPoint.getDateTime();
            heartRateDateTime   =heartRatePoint.getDateTime();
            
            // If a change of track occurs: reset the average
            if (trackPoint.pointIsStartOfTrack)
            {
                sum=0.0;
                count=0;
            }
            
            if (heartRateDateTime.lteq(trackPointDateTime))
            {
                // heart rate values before or at the trackpoint: sum and count
                if (heartRatePoint.getHeartRateValue()>0)
                {
                    sum+=heartRatePoint.getHeartRateValue();
                    count++;
                }
                if (heartRateIterator.hasNext())
                {
                    heartRatePoint=heartRateIterator.next();
                }
                else
                {
                    heartRatePoint=null;
                    exit=true;
                }
            }
            else
            {
                // add average heartrate to trackpoint
                if (count>0)
                {
                    trackPoint.setHeartRate((int)(sum/count));
                    sum     =0.0;
                    count   =0;
                }
                else
                {
                    trackPoint.setHeartRate(-1);
                }

                if (trackLogIterator.hasNext())
                {
                    trackPoint=trackLogIterator.next();
                }
                else
                {
                    trackPoint=null;
                    exit=true;
                }
            } 
        }
        // If a heart rate average and a track point remaining, 
        // add the average to the trackpoint
        if (count>0 && trackPoint!=null)
        {
            trackPoint.setHeartRate((int)(sum/count));
        }

        // Set heart rate of any trackpoint left to -1
        while (trackLogIterator.hasNext())
        {
            trackPoint=trackLogIterator.next();
            
            trackPoint.setHeartRate(-1);
        }        

    }
    
    
    
    
    /**
     * This method returns the id of the segment
     * @return The id of the segment
     */
    public int getSegmentId()
    {
        return this.trackSegmentNo;
    }
    
    /**
     * This method returns the number of waypoints.
     * @return The number of waypoints
     */
    public int getNumberOfWaypoints()
    {
        return this.wayPoints.size();
    }

    /**
     * This method returns the number of trackpoints.
     * @return The number of trackpoints
     */
    public int getNumberOfTrackpoints()
    {
        return this.trackPoints.size();
    }

    /**
     * This method returns the number of heartrate points.
     * @return The number of heartratepoints
     */
    public int getNumberOfHeartRatePoints()
    {
        return this.heartRatePoints.size();
    }
    
    
    /**
     * This method returns whether this segment is empty, i.e. contains no
     * data
     * @return True when empty, false if any data present 
     */
    public boolean isEmptySegment()
    {
        boolean isEmpty;
        
        if (this.wayPoints.size()>0 || 
            this.trackPoints.size()>0 || 
            this.heartRatePoints.size()>0)
        {
            isEmpty=false;
        }
        else
        {
            isEmpty=true;
        }
        return isEmpty;
    }

    /**
     * This method returns the array of track points.
     * @return The ArrayList with track points in this segment
     */
    public ArrayList<TrackLogPoint> getTrackPoints()
    {
        return this.trackPoints;
    }
    
    /**
     * This method returns the array with waypoints
     * @return The ArrayList with waypoints
     */
    public ArrayList<TrackLogPoint> getWayPoints()
    {
        return this.wayPoints;
    }
    
    /**
     * This method returns the array with heart rate values
     * @return The ArrayList with heart rate values
     */
    public ArrayList<HeartRatePoint> getHeartRatePoints()
    {
        return this.heartRatePoints;
    }
    
    /**
     * This method returns the start time in UTC of the segment. This is
     * the time of the 1st data point in this segment
     * @return The start time of the segment, or null if no data present.
     */
    public DateTime getStartTime()
    {
        DateTime dateTime;
        DateTime pointDateTime;
        
        dateTime=null;
        
        if (this.trackPoints.size()>0)
        {
            dateTime=trackPoints.get(0).getDateTime();
        }
        
        if (this.wayPoints.size()>0)
        {
            pointDateTime=wayPoints.get(0).getDateTime();
            
            if (dateTime==null)
            {
                dateTime=pointDateTime;
            } 
            else if (pointDateTime.lt(dateTime))
            {
                dateTime=pointDateTime;
            }
        }
        
        if (this.heartRatePoints.size()>0)
        {
            pointDateTime=heartRatePoints.get(0).getDateTime();
            if (dateTime==null)
            {
                dateTime=pointDateTime;
            } 
            else if (pointDateTime.lt(dateTime))
            {
                dateTime=pointDateTime;
            }
        }
        
        return dateTime;
    }
    
    /**
     * This method returns the end time of the segment as UTC time.
     * This is the datetime of the last data point in the segment
     * @return The end time of the segment or null if the segment is empty
     */
    public DateTime getEndTime()
    {
        DateTime dateTime;
        DateTime pointDateTime;
        
        dateTime=null;
        
        if (this.trackPoints.size()>0)
        {
            dateTime=trackPoints.get(trackPoints.size()-1).getDateTime();
        }
        
        if (this.wayPoints.size()>0)
        {
            pointDateTime=wayPoints.get(wayPoints.size()-1).getDateTime();
            if (dateTime==null)
            {
                dateTime=pointDateTime;
            } 
            else if (pointDateTime.gt(dateTime))
            {
                dateTime=pointDateTime;
            }
        }
        
        if (this.heartRatePoints.size()>0)
        {
            pointDateTime=heartRatePoints.get(heartRatePoints.size()-1).getDateTime();
            if (dateTime==null)
            {
                dateTime=pointDateTime;
            } 
            else if (pointDateTime.gt(dateTime))
            {
                dateTime=pointDateTime;
            }
        }
        
        return dateTime;
    }    

    /**
     * This method returns the distance traveled in the segment
     * @return The distance in meters
     */
    public double getSegmentDistance()
    {
        double                  distance;
        double                  radius;
        double                  lat1, lat2, lon1, lon2;
        double                  dLat, dLon, c, a;
        Iterator<TrackLogPoint> iterator;
        TrackLogPoint           point;
        TrackLogPoint           prevPoint;
        
        distance=0.0;
        radius=6371000;
        
        iterator=trackPoints.iterator();
        
        if (iterator.hasNext())
        {
            prevPoint=iterator.next();

        
            while (iterator.hasNext())
            {
                point=iterator.next();

                lat1=2*Math.PI*prevPoint.getLatitude()/360.0;
                lon1=2*Math.PI*prevPoint.getLongitude()/360.0;

                lat2=2*Math.PI*point.getLatitude()/360.0;
                lon2=2*Math.PI*point.getLongitude()/360.0;


                // Spherical law of cosines
                distance+= Math.acos(Math.sin(lat1)*Math.sin(lat2) + 
                           Math.cos(lat1)*Math.cos(lat2) *
                           Math.cos(lon2-lon1)) * radius;

/*
                
                // Haversine method
                dLat = lat2-lat1;
                dLon = lon2-lon1;

                a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                    Math.sin(dLon/2) * Math.sin(dLon/2) * Math.cos(lat1) * Math.cos(lat2); 
                c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a)); 
                distance += radius * c;                
*/                
                prevPoint=point;
            }
        }
        
        return distance;
    }
    
    /**
     * This method returns the max speed logged by the device
     * @return The max speed in m/s.
     */
    public double getSegmentMaxSpeed()
    {
        double                  speedMax;
        double                  speed;
        Iterator<TrackLogPoint> iterator;
        TrackLogPoint           point;
        
        speedMax=0.0;
        
        iterator=this.trackPoints.iterator();
        while (iterator.hasNext())
        {
            point=iterator.next();
            speed=point.getSpeed();
            
            if (speed>speedMax)
            {
                speedMax=speed;
            }
        }
        
        return speedMax;
    }
    
    /**
     * This method returns the duration of the segment. It is the time difference
     * between last data point and first datapoint
     * @return Duration in seconds
     */
    public long getSegmentDuration()
    {
        DateTime    start;
        DateTime    end;
        long        duration;
        
        duration=0;
        
        start=getStartTime();
        end  = getEndTime();
        
        if (start!=null && end!=null)
        {
            duration=start.numSecondsFrom(end);
        }
        
        return duration;
    }
    
    /**
     * This method calculates the calories burned during the segment by parsing
     * all track points. If a heart rate value is available, this is used 
     * to estimate the calories. If not, the metabolic equivalent is used.
     * @return The energy burned in kCalories.
     */
    public double getSegmentCalories()
    {
        Settings                    settings;
        Iterator<TrackLogPoint>     iterator;
        double                      age;
        double                      length;
        double                      weight;
        String                      gender;
        double                      restMetabolicRate;
        double                      calories;
        double                      burned;
        double                      metabolicEquivalent;
        double                      rate;
        TrackLogPoint               point;
        TrackLogPoint               prevPoint;
        DateTime                    time;
        DateTime                    prevTime;
        double                      seconds;
        boolean                     isMale;
        
        calories=0.0;
        
        settings            =Settings.getInstance();
        age                 =settings.getProfileAge();
        length              =settings.getProfileLength();
        weight              =settings.getProfileWeight();
        gender              =settings.getProfileGender(); 
        metabolicEquivalent =settings.getMetabolicEquivalent();
             
        if (gender.equals("male"))
        {
            isMale=true;
        }
        else
        {
            isMale=false;
        }
/*        
        // Harris-Benedict Equation (1919) for basal metabolic rate
        // In kCal/day
        if (isMale)
        {
            restMetabolicRate=13.7516*weight+5.0033*length-6.7550*age+66.4730;
        }
        else
        {
            restMetabolicRate=9.5634*weight+1.8496*length-4.6756*age+665.0955;
        }
*/        
        // Mifflin St. Jeor Equation (1990) for basal metabolic rate
        // In kCal/day
        if (isMale)
        {
            restMetabolicRate=10.0*weight+6.25*length-5.0*age+5.0;
        }
        else
        {
            restMetabolicRate=10.0*weight+6.25*length-5.0*age-161.0;
        }
        
        iterator=this.trackPoints.iterator();
        if  (iterator.hasNext())
        {
            prevPoint=iterator.next(); 
            while (iterator.hasNext())
            {
                point       =iterator.next();
                time        =point.getDateTime();
                prevTime    =prevPoint.getDateTime();
                seconds     =prevTime.numSecondsFrom(time);
                rate        =point.getHeartRate();
                
                if (rate>0)
                {
                    // age in year, weigth in kg, heart rate in bpm, duration in minutes -> result kCal

                    if (isMale)
                    {
                        burned = ( (age * 0.2017) + (weight * 0.1988)+ (rate * 0.6309) - 55.0969) * (seconds/60.0) / 4.184;
                    }
                    else
                    {
                        burned = ( (age * 0.074) + (weight * 0.1263) + (rate * 0.4472) - 20.4022) * (seconds/60.0) / 4.184;       
                    }
                }
                else
                {
                    burned=metabolicEquivalent*restMetabolicRate/24*seconds/3600;
                }
                calories+=burned;
                
                prevPoint=point;      
            }
        }        
        
        
        return calories;
    }
    
    
}
