/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
/**
 * This class represents the cache for the track log memory. By storing 
 * already downloaded blocks on disk and use it.
 * The implementation assumes the blocks are written in the order oldest to 
 * newest. It is not allowed to write data randomly in cache.
 * @author Jorgen
 */
public class TrackMemoryCache 
{
    private static final int    NO_BLOCK_WRITTEN=-1;
    
    int startBlock;             // absolute number of first flash block memory to store
    int endBlock;               // absolute number of last flash block memory to store
    int blockSize;              // number of bytes in a flash memory block
    int serial;                 // serial number of the device to cache
    String device;               // type of the device to cache
    byte byteBlock[][];         // the cache
    
    int lastBlockWritten;       // absolute number of the last flash block written
    boolean isCacheWrappedAround;    // indicates whether the memory has looped around
    
    boolean isUpdated;          // Indicates whether the cache has been updated since last write to file
    
    Settings settings;
    
    public TrackMemoryCache(int startBlock, int endBlock, int blockSize, String device, int serial)
    {
        int i;
        
        DebugLogger.info("Create cache for device: "+device+" serial: "+serial);

        settings=Settings.getInstance();
        
        this.startBlock =startBlock;
        this.endBlock   =endBlock;
        this.blockSize  =blockSize;
        this.device     =device;
        this.serial     =serial;
        byteBlock       =new byte[endBlock-startBlock+1][blockSize];
          
        resetCache();
        readCache();
        
        isUpdated=false;    // cache is empty, so no need to write to file
        
    }

    /**
     * Reset the cache.
     */
    public final void resetCache()
    {
        // This effectively clears the cache without 'erasing' it
        this.lastBlockWritten=NO_BLOCK_WRITTEN;
        this.isCacheWrappedAround=false;
    }
    
    
    /**
     * Returns if the cache corresponds to indicated device and serial
     * @param device Device type
     * @param serial Device serial
     * @return True if the cache corresponds to the device and serial, false 
     *         if not.
     */
    public boolean checkCache(String device, int serial)
    {
        boolean isOk;
        if ((this.device.equals(device)) && (this.serial==serial))
        {
            isOk=true;
        }
        else
        {
            isOk=false;
        }
        return isOk;
    }

    /**
     * This method validates the cache, checks its content corresponds 
     * to the device track flash memory. It is required, since the track memory
     * might have been erased
     * @param oldestData Flash block data containing the oldest data
     * @param blockNumber Number of the flash block 
     */
    public void validateCache(byte[] oldestData, int blockNumber)
    {
        int     i;
        boolean isEqual;

        DebugLogger.info("Validating cache based on block 0x"+Integer.toHexString(blockNumber));
        // We check the date-time stamp of the 1st record, which is the oldest
        // record in the block.
        i=1;
        isEqual=true;
        while (i<6 && isEqual)
        {
            if (byteBlock[blockNumber-startBlock][i]!=oldestData[i])
            {
                isEqual=false;
            }
            i++;
        }
        
        // If the data does not correspond, reset the cache
        if (!isEqual)
        {
            DebugLogger.info("Cache invalidated");
            resetCache();
        }
        else
        {
            DebugLogger.info("Cache validated");
        }
    }
    
    /**
     * This method returns the file name for this cache
     * @return 
     */
    private String getFileName()
    {
        String fileName;
        String path;
        
        path=settings.getLogPath();                         // get log path. Always ends with /
        fileName=path+"cache_"+device+"_"+serial+".bin";
        return fileName;
    }
    

    /**
     * This method reads the cache from disk, if it exists
     */
    private void readCache()
    {
        String      fileName;
        File        file;
        ObjectInputStream input;
        byte[]      result;
        int         dummy;
        int         bytesRead;
        int         totalBytesRead;
        int         bytesRemaining;
        int         i;
        
        fileName=getFileName();
        file = new File(fileName);
        
        // Check if the file exists. If so, read it.
        if (file.exists())
        {
        
            DebugLogger.info("Reading cache binary file " + fileName);
            
            DebugLogger.debug("File size: " + file.length());
            input = null;
            try 
            {
              input = new ObjectInputStream(new BufferedInputStream(new FileInputStream(file)));

              startBlock=input.readInt();
              endBlock=input.readInt();
              blockSize=input.readInt();
              lastBlockWritten=input.readInt();
              dummy=input.readInt();
              if (dummy>0)
              {
                  this.isCacheWrappedAround=true;
              }
              else
              {
                  this.isCacheWrappedAround=false;
              }

              i=0;
              while (i<endBlock-startBlock+1)
              {
                  totalBytesRead = 0;
                  while(totalBytesRead < blockSize)
                  {
                    bytesRemaining = blockSize - totalBytesRead;
                    //input.read() returns -1, 0, or more :
                    bytesRead = input.read(byteBlock[i], totalBytesRead, bytesRemaining); 
                    if (bytesRead > 0)
                    {
                      totalBytesRead += bytesRead;
                    }
                  }
                  if (totalBytesRead!=blockSize)
                  {
                     DebugLogger.error("Error reading cache: block "+i+" has size 0x"+Integer.toHexString(totalBytesRead));
                  }
                  DebugLogger.debug("Num bytes read: " + totalBytesRead);

                  i++;
              }
              /*
               the above style is a bit tricky: it places bytes into the 'result' array; 
               'result' is an output parameter;
               the while loop usually has a single iteration only.
              */
              input.close();

            }
            catch (FileNotFoundException ex) 
            {
              DebugLogger.error("Cache file not found.");
              resetCache();
            }
            catch (IOException ex) 
            {
              DebugLogger.error("IO Exception reading cache file: "+ex.toString());
              resetCache();
            }
        }
        else
        {
            // Reset the cache
            resetCache();
            DebugLogger.debug("No cache file found");
        }
    }
    
    /**
     * This method writes the cache memory to file.
     */
    private void writeCache()
    {
        ObjectOutputStream      output;
        int                     i;
        String                  fileName;
        
        
        if (isUpdated)
        {
            fileName=getFileName();

            DebugLogger.info("Writing binary cache file "+fileName);
            try 
            {
              output = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(fileName)));
              output.writeInt(startBlock);
              output.writeInt(endBlock);
              output.writeInt(blockSize);
              output.writeInt(lastBlockWritten);
              if (this.isCacheWrappedAround)
              {
                  output.writeInt(1);
              }
              else
              {
                  output.writeInt(0);
              }

              i=0;
              while (i<endBlock-startBlock+1)
              {
                  output.write(byteBlock[i]);
                  i++;
              }
              output.close();
            }
            catch(FileNotFoundException ex)
            {
              DebugLogger.error("Cannot write cache file " + fileName + ". Perhaps 'logPath' directory does not exist");
            }
            catch(IOException ex)
            {
              DebugLogger.error("IO exception while writing cache file: "+ex.toString());
            }        
            
            isUpdated=false;
        }
    }
    
    
    /**
     * This method returns the cached block indicated or null if it has not  
     * been cached before.
     * @param blockNumber Absolute flash block to retrieve
     * @param isSourceWrappedAround Indicates whether the track log memory is wrapped around
     * @param isRecentData In case the track log has been wrapped around, this
     *                     indicates the data is assumed to be after wrap around
     *                     (true) or from prior to the wrap around (false) 
     * @return True if the data was read from cache, false if not
     */
    public boolean getBlock(byte[] target, int blockNumber, int mostRecentTrackMemBlock)
    {
        boolean bytesRead;
        
        // Default value: no data read
        bytesRead=false;
/*        
        // Check if the track log has wrapped around. If so, all blocks
        // in the cache are filled with data.
        if (isWrappedAround)
        {
            if (isRecentData)
            {
                // if data from after the wrap around is requested, it has only
                // be cached from startBlock to and including lastBlockWritten.
                // Any data after lastBlockWritten is 'old' data from prior
                // to the wrap around
                // Note: lastBlockWritten cannot be trusted, since in cache it  
                //       might be incomplete, whereas in device memory it may be 
                //       complete. Therefore we treat it as not written.
                if (blockNumber<lastBlockWritten)
                {
                    bytesRead=true;
                }
            }
            else
            {
                if (lastBlockWritten!=NO_BLOCK_WRITTEN)
                {
                    bytesRead=true;
                }
            }
        }
        else
        {
            // The track log has not wrapped around. This means 
            // the cache has been filled from startBlock to and including 
            // lastBlockWritten. Other blocks are empty.
            // Note: lastBlockWritten cannot be trusted, since in cache it might
            //       be incomplete, whereas in device memory it may be complete
            
            if (blockNumber<lastBlockWritten)
            {
                bytesRead=true;
            }
        }
        
*/
/*        
        if (!isCacheWrappedAround)
        {
            if (!isSourceWrappedAround)
            {
                // * isRecentData
                // # !isRecentData
                // L lastBlockWritten
                // V valid cache value
                // Cache |*******************L             |
                // Track |*******************NNNNNN        |
                // Valid |VVVVVVVVVVVVVVVVVVV              |
                // The track log has not wrapped around. This means 
                // the cache has been filled from startBlock to and including 
                // lastBlockWritten. Other blocks are empty.
                // Note: lastBlockWritten cannot be trusted, since in cache it might
                //       be incomplete, whereas in device memory it may be complete

                if (blockNumber<lastBlockWritten)
                {
                    bytesRead=true;
                }                
            }
            else
            {
                // Cache |*******************L             |
                // Track |***L#############################|
                // Valid |    VVVVVVVVVVVVVVV              |
                // The Cache is not wrapped around, whereas the track memory has
                // Recent data is not in the cache. Non recent data is.
                // We force a write for blockNumber=startBlock, to wrap around the cache
                
                if (!isRecentData && blockNumber<lastBlockWritten && blockNumber>this.startBlock)
                {
                    bytesRead=true;
                }
            }
        }
        else
        {
            if (!isSourceWrappedAround)
            {
                // Cache |***L#############################|
                // Track |*******************L             |
                // Valid |                                 |
                // The cache is wrapped around, while the tracklog is not
                // Should not occur...
                DebugLogger.error("Chache error");
            }
            else
            {
                // Cache |*******************L#############|
                // Track |***********************L#########|
                // Valid |                        VVVVVVVVV|
                if (!isRecentData)
                {
                    if (blockNumber>this.lastBlockWritten)
                    {
                        bytesRead=true;
                    }
                }
                else
                {
                    if (blockNumber<this.lastBlockWritten)
                    {
                        bytesRead=true;
                    }                    
               
                }
                
            }
        }
*/
        // Check if the track memory is wrapped around. When this happens
        // the most recent data block in memory is smaller than the last block
        // written
        if (mostRecentTrackMemBlock<this.lastBlockWritten)
        {
            if (blockNumber>mostRecentTrackMemBlock && blockNumber<this.lastBlockWritten)
            {
                bytesRead=true;
            }
        }
        // Not wrapped around
        else if (mostRecentTrackMemBlock>this.lastBlockWritten)
        {
            // Data stores after the wrap around or, if not wrapped around.
            if (blockNumber<this.lastBlockWritten)
            {
                bytesRead=true;
            }
            
            // Data stored before the wrap around
            if (blockNumber>mostRecentTrackMemBlock)
            {
                bytesRead=true;
            }
        }
        // Tracklog corresponds to cache (however most recent block may differ)
        // Whether it is wrapped around does not matter
        else
        {
            if (blockNumber!=this.lastBlockWritten)
            {
                bytesRead=true;
            }
        }
        
        if (bytesRead)
        {
            System.arraycopy(byteBlock[blockNumber-startBlock], 0, target, 0, blockSize);
            DebugLogger.info("Read block 0x"+Integer.toHexString(blockNumber)+" from cache");
        }
        
//System.out.println("Blocknumber "+blockNumber+" source "+isSourceWrappedAround+" cache "+this.isCacheWrappedAround+" result "+bytesRead);        
//System.out.println("Blocknumber "+blockNumber+" most recent "+mostRecentTrackMemBlock+" lbw "+this.lastBlockWritten+" result "+bytesRead);        
        
        return bytesRead;
    }
    
    /**
     * This method writes data to the cache.
     * @param blockNumber The absolute flash block to write
     * @param blockData The data to write
     * @param isRecentData Indicates whether this is old (before wrap around)
     *                     data (false) or not true
     */
    public void writeBlock(int blockNumber, byte[] blockData)
    {
        int i;
        
        DebugLogger.info("Write block 0x"+Integer.toHexString(blockNumber)+" to cache");

        System.arraycopy(blockData, 0, byteBlock[blockNumber-startBlock], 0, blockSize);
        
        isUpdated=true;
 
/*        
        // If the first block is written somewhere
        if ((lastBlockWritten==NO_BLOCK_WRITTEN) && (blockNumber!=startBlock))
        {
            
        }
        
        // We assume blocks are written in the order from oldest to most recent.
        // Blocks are written subsequently. Therefore we can detect whether the
        // tracklog wraps around
        if ((lastBlockWritten==endBlock) &&(blockNumber==startBlock))
        {
            isWrappedAround=true;
        }
  */
        // If we write non recent data, the tracklog is wrapped
        if (/*(!isRecentData)*/(blockNumber==this.startBlock) && (this.lastBlockWritten==this.endBlock))
        {
            isCacheWrappedAround=true;
            lastBlockWritten=blockNumber;
        }
        else
        {
            // Only if recent data is written, update lastBlockWritten
            // So it always indicates the separation between recent data
            // and old data (for wrap)
            lastBlockWritten=blockNumber;
        }
    }
    
    /**
     * This method writes the cache from memory to file.
     */
    public void makeCachePersistent()
    {
        this.writeCache();
    }

    
    /**
     * This method checks the content of indicated cached block by comparing
     * the contents to the data provided.
     * @param data The data to verify against
     * @param block The absolute page number.
     * @return True if identical, false if cache page not identical
     */
    public boolean checkBlockContent(byte[] data, int block)
    {
        boolean isOk;
        int i;
        
        isOk=true;
        
        // Simply check the contents
        i=0;
        while ((i<this.blockSize) && !isOk)
        {
            if (this.byteBlock[block][i]!=data[i])
            {
                isOk=false;
            }
                
            i++;
        }
        
        return isOk;
    }
        
    /**
     * This method checks if the lastBlockWritten parameter has indicated
     * value
     * @param block The absolute number of the page that has been written
     *              as last during track download.
     * @return True if ok, false if not ok (lastBlockWritten has not the 
     *         indicated value) 
     */
    public boolean checkLastBlockWritten(int block)
    {
        boolean isOk;
        
        isOk=(lastBlockWritten==block);

        return isOk;
    }
    
}
