/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import hirondelle.date4j.DateTime;

/**
 * This class represents a track point. It may be a track log point
 * (logged coordinate) or a waypoint (log triggerd by the user)
 * @author Jorgen van der Velde
 */
public class TrackLogPoint extends Coordinate
{
    public static final int         TRACKLOGPOINT_RECORDSIZE=0x20;

    
    /** Flags     */
    int                 flags;

    /** Date time stamp */
    DateTime            dateTime;

    /** Unknown */
    int                 status;

    /** Estimated horinzontal positioning error in cm */
    int                 ehpe;

    /** Not used */
    int                 satelliteMap;

    /** Elevation in cm. Measured by pressure sensor on the pro version */
    int                 elevationPressure;

    /** Elevation in cm. Measured by GPS */
    int                 elevationGps;


    /** speed in cm/s */
    int                 speed;

    /** course in 10e-2 degrees */
    int                 course;

    /** unkonwn */
    int                 timeout;
  
    /** unknown */
    int                 maxFound;

    /** Number of satellites */
    int                 satelliteNumber;


    /** */
    int                 weightCriteria;

    /** */
    int                 sleepTime;

    /** Device type. It is used to interpret the data  passed */
    Device.DeviceType   deviceType;

    /** Indicates whether this point is the 1st of the track */
    boolean             pointIsStartOfTrack;
    
    /** Indicates whether this point is the last of the track. May be not used!! */
    boolean             pointIsEndOfTrack;
    
    
    /** Heartrate, supported by the GT-820 and GT-900 */
    int                 heartRate;
    
    /** Cadence, supported by the GT-820 and GT-900 */
    int                 cadence;



    /**
     * Constructor
     * @author Jorgen van der Velde
     */
    public TrackLogPoint(byte[] record, Device.DeviceType deviceType)
    {
        DateTime            now;
        int                 yearNow;
        int                 year;
        int                 yearMod16;
        int                 month;
        int                 day;
        int                 hour;
        int                 minute;
        int                 millis;
        int                 second;
        int                 nanosecond;
        int                 recordPrevMillis;

        // Default values
        pointIsEndOfTrack  =false;
        pointIsStartOfTrack=false;
        
        // deviceType
        this.deviceType=deviceType;
        
        // Flags
        flags=record[0]&0xff;
       
        dateTime=ToolBox.bytesToDateTime(record[1], record[2], record[3], record[4], record[5]);

        // Status
        status=(record[0x06]&0xf0)>>4;

        // ehpe - expected horizontal position error
        ehpe=(record[0x06]&0xf0)<<4 | (record[0x07]&0xff);

        // satellite map. Apparently the bytes 08-0B are elevation on GT-800 Pro
/*
        satelliteMap=((record[0x08]&0xff)<<24) |
                     ((record[0x09]&0xff)<<16) |
                     ((record[0x0A]&0xff)<<8) |
                     ((record[0x0B]&0xff));
*/
        // latitude
        latitude    =(record[0x0C]<<24) |
                     ((record[0x0D]&0xff)<<16) |
                     ((record[0x0E]&0xff)<<8) |
                     (record[0x0F]&0xff);

        // longitude
        longitude    =(record[0x10]<<24) |
                     ((record[0x11]&0xff)<<16) |
                     ((record[0x12]&0xff)<<8) |
                     (record[0x13]&0xff);

        // elevation
        elevationPressure    =(record[0x08]<<24) |
                              ((record[0x09]&0xff)<<16) |
                              ((record[0x0a]&0xff)<<8) |
                              ((record[0x0b]&0xff));

        elevationGps =(record[0x14]<<24) |
                      ((record[0x15]&0xff)<<16) |
                      ((record[0x16]&0xff)<<8) |
                      ((record[0x17]&0xff));

        // Based on the device type choose the best elevation value
        if ((deviceType==Device.DeviceType.DEVICETYPE_GT800PRO) ||
            (deviceType==Device.DeviceType.DEVICETYPE_GT820PRO) ||
            (deviceType==Device.DeviceType.DEVICETYPE_GT900PRO))   
        {
            elevation=elevationPressure;
        }
        else
        {
            elevation=elevationGps;
        }


        // speed
        speed       =((record[0x18]&0xff)<<8) |
                     ((record[0x19]&0xff));

        // course
        course      =((record[0x1A]&0xff)<<8) |
                     ((record[0x1B]&0xff));

        // timeout
        timeout     =record[0x1C]&0xff;

        // Number of satellites????
        satelliteNumber=record[0x1D]&0x0f;
        
            

        
        // Set to unknown. Cadence must be set after creation
        cadence=-1;
        
        // Set to unknown. Heartrate must be set after creation
        heartRate=-1;
        
        
        
        
    }
    

    

    /**
     * Print the info in a string
     * @author Jorgen van der Velde
     */
    @Override
    public String toString()
    {
       String outputString;

       outputString=new String("");

       return outputString;
    }

    /**
     * Get the timestamp of the record
     * @return Date-time
     * @author Jorgen van der Velde
     */
    public DateTime getDateTime()
    {
        return dateTime;
    }



    /**
     * Get ehpe in m
     * @return ehpe in meters
     * @author Jorgen van der Velde
     */
    public double getEhpe()
    {
        return ((double)ehpe)/1e2*16.0;
    }


    /**
     * Get elevation measured by atmospheric pressure (GT-800 Pro)
     * @return Elevation in meter
     * @author Jorgen van der Velde
     */
    public double getElevationByAtmosphericPresure()
    {
        return ((double)this.elevationPressure)/1e2;
    }

    /**
     * Get elevation
     * @return Elevation in meter
     * @author Jorgen van der Velde
     */
    public double getElevationByGps()
    {
        return ((double)this.elevationGps)/1e2;
    }



    /**
     * Returns whether the datapoint marks the start of a track.
     * This point is the 1st point after the GT-800 device is 
     * switched on. Does not work for GT-820 and GT-900.
     * @return True if it is the start of a track, false if not
     * @author Jorgen van der Velde
     */
    public boolean isStartOfTrack()
    {
        return ((flags&0x40)!=0x00);    
    }

    /**
     * Returns whether the datapoint marks the end of a track. The
     * end of a track is recorded if the GT-800 device is switched off.
     * Does not work for GT-820 and GT-900.
     * @return True for the endpoint, false if not
     * @author Jorgen van der Velde
     */
    public boolean isEndOfTrack()
    {
        return ((flags&0x20)!=0x00);
    }

    /**
     * This method indicates if current track point is a waypoint logged
     * by the user (101-999)
     * @return True if a waypoint, false for a regular track point
     */
    public boolean isWaypoint()
    {
        return ((flags&0x04)!=0x00);
    }

    /**
     * This method returns the course in 0-360 degrees.
     * @return The course.
     */
    public double getCourse()
    {
        return (double)course/100.0;
    }

    /**
     * This method returns the speed at this track point
     * @return The speed in m/s
     */
    public double getSpeed()
    {
        return (double)speed/100.0;
    }

    /**
     * This method returns the number of satellites
     * @return The number of satellites.
     */
    public int getSatelliteNumber()
    {
        return satelliteNumber;
    }

    /**
     * This method returns the device type of the device that logged this point.
     * @return The device type
     */
    public Device.DeviceType getDeviceType()
    {
        return deviceType;
    }
    
    /**
     * This method sets the heartrate value
     * @param rate The heartrate in bpm. -1 if undefined.
     */
    public void setHeartRate(int rate)
    {
        this.heartRate=rate;
    }

    /**
     * This method returns the heart rate
     * @return The heart rate in bpm or -1 if not defined.
     */
    public int getHeartRate()
    {
        return this.heartRate;
    }
    
    /**
     * This method sets the biking cadence
     * @param cadence New cadence value in rpm, -1 if not defined
     */
    public void setCadence(int cadence)
    {
        this.cadence=cadence;
    }
    
    /**
     * This method returns the biking cadence
     * @return Cadence value in rpm or -1 if not defined
     */
    public int getCadence()
    {
        return this.cadence;
    }
    
}


