/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import java.util.ArrayList;
import java.util.Iterator;
import hirondelle.date4j.DateTime;


/**
 *  This class represents the Track Log. It contains functionality to turn
 *  downloaded device data into tracks. It discerns track points (points 
 *  defining the logged track), waypoints (user logged point-of-interests) and
 *  heart rate values (when heart rate monitor is attached, GT-820 and GT-900)
 *  It offers functionality to present the tracks and track related information.
 *  @author Jorgen van der Velde
 */
public class TrackLog
{
    // Some administration for the parsing
    private static boolean              counterResetFound=false;
    private static int                  prevMillis=0xffff;
    private static TrackLogPoint        previousPoint=null;
    
    
    /** Length of a record representing a trackpoint/waypoint/log entry in the device data */
    private static int                  RECORDLENGTH=0x20;

    /** The one and only instance of this class */
    static TrackLog                     theInstance=null;

    /** Helper, containing one log record */
    private byte[]                      record;
    

    /** The tracks. In fact its represents a subdivision of the points in 
     *  the log in tracks and track segments. It consists of a list of 
     *  pointers.
     */
    private ArrayList<Track>            tracks;

    /** Type of the device that logged the tracks */
    private Device.DeviceType           deviceType;
    
    /** Current track under construction, when appending data */
    private Track                       currentTrack;
    
    private boolean                     stateIsLogging;
    
    
    /**
     * Constructor
     * @author Jorgen van der Velde
     */
    private TrackLog()
    {
        record      =new byte[RECORDLENGTH];
        tracks      =new ArrayList();
    }

    /**
     * Get the one and only singleton instance of this class
     * @author Jorgen van der Velde
     */
    public static TrackLog getInstance()
    {
        if (theInstance==null)
        {
            theInstance=new TrackLog();
        }
        return theInstance;
    }


    /**
     * Convert the data to trackpoints, waypoints and heart rate values
     * and append these to the tracks. Prior to start appending data 
     * initialiseTrackLog() must be called. Multiple data blocks must be appended
     * in sequential order. After all blocks are appended, finishTrackLog()
     * must be called.
     * @param data The datablock
     * @param length Length of the datablock. Must be multiple of 0x20
     * @author Jorgen van der Velde
     */
    public void appendData(byte[] data, int length)
    {
        int                         records;
        int                         i;
        TrackLogPoint               point;
        ArrayList<HeartRatePoint>   newHeartRatePoints;
        int                         flags;
        int                         millis;
        int                         recordPrevMillis;

        byte[]                      logBytes;
        byte                        nextByte;
        String                      logString;
        
        boolean                     trackStartFound;
        
        
        if (length%RECORDLENGTH!=0)
        {
            DebugLogger.error("Datalenght not a multiple of 0x20 bytes");
        }
        else
        {
            records=length/RECORDLENGTH;
            i=0;
            while (i<records)
            {
                /*********************************************************************************\
                 * Split up in track/waypoint records
                 \********************************************************************************/
                
                trackStartFound=false;
                
                
                // Copy part of the data to the record 
                ToolBox.copyBytes(data, record, i*RECORDLENGTH, RECORDLENGTH);
//                System.out.println("Record "+i+" "+String.format("%02x", record[0])+" "+String.format("%02x", record[1]));
                
                // Extract minimum amount of data from the record, needed
                // for parsing
                // The flags
                flags=(record[0]&0xff);

                // seconds as milliseconds
                millis=((record[4]&0xff)<<8)|(record[5]&0xff);        

                // Last two bytes are milliseconds from previous record
                recordPrevMillis=((record[30]&0xff)<<8)|(record[31]&0xff);

                /*********************************************************************************\
                 * Find the start of track and add the records to the track or waypoint log.
                 * Detection of track boundaries depend on the type of device
                 \********************************************************************************/
                
                if ((deviceType==Device.DeviceType.DEVICETYPE_GT820) || 
                    (deviceType==Device.DeviceType.DEVICETYPE_GT820PRO) ||
                    (deviceType==Device.DeviceType.DEVICETYPE_GT900) || 
                    (deviceType==Device.DeviceType.DEVICETYPE_GT900PRO))
                {
                    // flags=0xF1: the record is a debug log record
                    // for GT-820 and GT-900 it is used to detect the track 
                    // boundaries
                    // power down            : !!! SYSTEM OFF, RESET COUNTER, POWER DOWN
                    // power up              : POWER UP
                    // reset counter         : RESET COUNTER
                    // connect/disconnect USB: RESET COUNTER, POWER DOWN, POWER UP
                    // device reset          : RESET COUNTER, RESET COUNTER

                    if (flags==0xF1)
                    {
                                                // For the GT-820 and GT-900: find the 'RESET COUNTER' message
                        logBytes=new byte[24];            

                        ToolBox.copyBytes(record, logBytes, 6, 24);

                        logString=new String(logBytes);
                        if (logString.startsWith("RESET COUNTER"))
                        {
                            
                                if (stateIsLogging)
                                {
                                    newTrack();
                                }
                        }
                        else if (logString.startsWith("POWER UP"))
                        {
                            if (!stateIsLogging)
                            {
                                newTrack();
                            }
                            stateIsLogging=true;
                        }
                        else if (logString.startsWith("!!!SYSTEM OFF"))
                        {
                            stateIsLogging=false;
                        }

                    }
                    // flags=0xF5: the record is a heart rate log record
                    // This record occurs once every 24 seconds when heart rate
                    // is monitored and contains 24 values (once per second)
                    // Convert and add to the heart rate log                
                    else if (flags==0xF5)
                    {
                        newHeartRatePoints=HeartRatePoint.parseHeartRateRecord(record);
                        currentTrack.appendHeartRates(newHeartRatePoints);
                    }
                    else if ((flags&0x10)==0)
                    {

                        point=new TrackLogPoint(record, deviceType);   
                        if (point.isWaypoint())
                        {
                            currentTrack.appendWaypoint(point);
                        }
                        else
                        {
                            currentTrack.appendTrackpoint(point);
                        }                        
                    }
                }
                else if ((deviceType==Device.DeviceType.DEVICETYPE_GT800) || 
                        (deviceType==Device.DeviceType.DEVICETYPE_GT800PRO))
                {
                    if ((flags&0x10)==0)
                    {
                        // Next we are going to indicate whether the track point is the
                        // start of the track. For the GT-800 and GT-800PRO we use the 
                        // flags
                        if ((flags&0x40)!=0)
                        {
                            newTrack();
                        }                        
                        
                        point=new TrackLogPoint(record, deviceType);   
                        if (point.isWaypoint())
                        {
                            currentTrack.appendWaypoint(point);
                        }
                        else
                        {
                            currentTrack.appendTrackpoint(point);
                        }                        
                    }
                }
                else if ((deviceType==Device.DeviceType.DEVICETYPE_GT120))                    
                {
                    // Start of track
                    if ((flags&0x40)!=0)
                    {
                        newTrack();
                    }
                    else
                    {
                        // It has appeared that there are records with all bytes
                        // except the flags are 0. Simply check byte 1
                        if (((flags&0x10)==0) && record[1]!=0)
                        {
                            point=new TrackLogPoint(record, deviceType);   

                            if (point.isWaypoint())
                            {
                                currentTrack.appendWaypoint(point);
                            }
                            else
                            {
                                currentTrack.appendTrackpoint(point);
                            }
                        }
                    }
                }
             
                
                
                
                
                
                
/*                
                
                // flags=0xF1: the record is a debug log record
                // for GT-820 and GT-900 it is used to detect the track 
                // boundaries
                // power down            : !!! SYSTEM OFF, RESET COUNTER, POWER DOWN
                // power up              : POWER UP
                // reset counter         : RESET COUNTER
                // connect/disconnect USB: RESET COUNTER, POWER DOWN, POWER UP
                // device reset          : RESET COUNTER, RESET COUNTER
                
                if (flags==0xF1)
                {
                    if ((deviceType==Device.DeviceType.DEVICETYPE_GT820) || 
                        (deviceType==Device.DeviceType.DEVICETYPE_GT820PRO) ||
                        (deviceType==Device.DeviceType.DEVICETYPE_GT900) || 
                        (deviceType==Device.DeviceType.DEVICETYPE_GT900PRO))
                    {
                        // For the GT-820 and GT-900: find the 'RESET COUNTER' message
                        logBytes=new byte[24];            

                        ToolBox.copyBytes(record, logBytes, 6, 24);

                        logString=new String(logBytes);
                        if (logString.startsWith("RESET COUNTER"))
                        {
                            
                                if (stateIsLogging)
                                {
                                    newTrack();
                                }
                        }
                        else if (logString.startsWith("POWER UP"))
                        {
                            if (!stateIsLogging)
                            {
                                newTrack();
                            }
                            stateIsLogging=true;
                        }
                        else if (logString.startsWith("!!!SYSTEM OFF"))
                        {
                            stateIsLogging=false;
                        }
                    }

                }
                // flags=0xF5: the record is a heart rate log record
                // This record occurs once every 24 seconds when heart rate
                // is monitored and contains 24 values (once per second)
                // Convert and add to the heart rate log                
                else if (flags==0xF5)
                {
                    newHeartRatePoints=HeartRatePoint.parseHeartRateRecord(record);
                    currentTrack.appendHeartRates(newHeartRatePoints);
                }
                // flags=0x0X: the record is a track/waypoint
                else if ((flags&0x10)==0)
                {
                    // Next we are going to indicate whether the track point is the
                    // start of the track. For the GT-800 and GT-800PRO we use the 
                    // flags, for the other devices we use RESET COUNTER log
                    if ((deviceType==Device.DeviceType.DEVICETYPE_GT800) || 
                        (deviceType==Device.DeviceType.DEVICETYPE_GT800PRO))   
                    {
                        if ((flags&0x40)!=0)
                        {
                            newTrack();
                        }
                    }                    

                    
                    // On the GT-120 empty recoerds (all bytes 0 except the flags) occur
                    if (flags!=0xc2 && record[1]!=0)
                    {
//                        System.out.println("add");
                        point=new TrackLogPoint(record, deviceType);   

                        if (point.isWaypoint())
                        {
                            currentTrack.appendWaypoint(point);
                        }
                        else
                        {
                            currentTrack.appendTrackpoint(point);
                        }
                    }                    
                }                
*/                
                
                /*********************************************************************************\
                 * On the newer devices two bytes in each records represent the milliseconds
                 * of previous records.
                 \********************************************************************************/
                
                // Check on missing in between records. Only holds for the 
                // Later version GT800+
                if (prevMillis!=0xffff && 
                        ((deviceType==Device.DeviceType.DEVICETYPE_GT800) ||
                        (deviceType==Device.DeviceType.DEVICETYPE_GT800PRO) ||
                        (deviceType==Device.DeviceType.DEVICETYPE_GT820) || 
                        (deviceType==Device.DeviceType.DEVICETYPE_GT820PRO) ||
                        (deviceType==Device.DeviceType.DEVICETYPE_GT900) || 
                        (deviceType==Device.DeviceType.DEVICETYPE_GT900PRO)))
                    
                {
                  if (recordPrevMillis!=prevMillis)
                  {
                      DebugLogger.error("Missing record(s) detected while parsing waypoints");
                  }
                }
                prevMillis=millis;
                
                
                i++;
            }
        }
    }

    /**
     * This method intialises the tracklog. It clears any data and prepares
     * the housekeeping for appending data. IT MUST BE CALLED PRIOR TO APPENDING
     * DATA
     */
    public void initialiseTracklog()
    {
        // Clear the tracks
        this.tracks.clear();

        // reset the track housekeeping
        Track.reset();
        
        counterResetFound=false;
        prevMillis=0xffff;
        previousPoint=null;        
    
        // Create the 1st track
        currentTrack=new Track();
        
        stateIsLogging=false;
    }
    
    /**
     * This method fisishes the track log. It adds any pending track under
     * construction. IT MUST BE CALLED AFTER APPENDING DATA. IF NOT, DATA MAY
     * BE LOST
     */
    public void finishTracklog()
    {
        if (!currentTrack.isEmptyTrack())
        {
            currentTrack.finish();
            tracks.add(currentTrack);
        }
    }
    
    /**
     * This method finishes the track under construction and creates a new
     * track under construction.
     */
    private void newTrack()
    {
        if (!currentTrack.isEmptyTrack())
        {
            currentTrack.finish();
            tracks.add(currentTrack);
            currentTrack=new Track();
        }
    }
    
    
    /**
     * Dump the log to the DebugLogger. Make sure debug logging is on
     * @author Jorgen van der Velde
     */
    /*
    public void dump()
    {
        Iterator iterator;
        TrackLogPoint point;

        iterator=log.iterator();

        while(iterator.hasNext())
        {
            point=(TrackLogPoint)iterator.next();
            DebugLogger.info(point.toString());
        }
    }

     */
    
    
    /**
     * Return the number of records (trackpoints and waypoints) in the log
     * @return The number of records
     * @author Jorgen van der Velde
     */
    int getNumberOfRecords()
    {
        Iterator<Track> iterator;
        Track           track;
        int             records;
        
        records     =0;
        iterator    =tracks.iterator();
        
        while (iterator.hasNext())
        {
            track=iterator.next();
            records+=track.getNumberOfTrackPoints();
            records+=track.getNumberOfWaypoints();
        }
        return records;
    }

    
    
    /**
     * Parse the collection of track/way points and detect the tracks and
     * track segments. 
     * To detect track boundaries, the code looks at start-of-track points.
     * End-of-track points are not taken into account. End-of-tracks may be
     * missing, whereas there is always a start-of-track.
     * To detect track segment boundaries, the code checks for missing
     * periods of data, defined by the setting segmentSeparationLimit.
     * This occurs when fix is lost or the device is put away. In the latter
     * case it stops logging when it does not detect movement.
     */
/*    
    public void parse()
    {
        int             element;
        int             maxElement;
        TrackLogPoint   point;
        TrackLogPoint   prevPoint;
        int             recordCount;    // all records: waypoints+track
        int             trackPointCount;     // only the track records
        int             waypointCount;  // only the waypoint records
        long            seconds;
        DateTime        dateTime;
        DateTime        prevDateTime;
        DateTime        startDateTime;
        DateTime        endDateTime;
        int             startElement;
        Track           track;
        TrackSegment    segment;
        int             trackNo;
        int             trackSegmentNo;
        int             segmentSeparation;

        recordCount     =0;
        trackPointCount =0;
        waypointCount   =0;
        startElement    =0;
        track           =null;
        trackNo         =0;
        trackSegmentNo  =0;
        prevPoint       =null;
        startElement    =0;
        startDateTime   =null;
        endDateTime     =null;
        seconds         =0;

        
        // Adorn a heart rate to the track points
        parseHeartRates();
        
        segmentSeparation=Settings.getInstance().getSegmentSeparationLimit();
        
        // Clear the segment vector
        tracks.clear();

        // Create the 1st track
        track=new Track();
        
        // Parse all trackpoints
        maxElement=log.size();
        element=0;
        while(element<maxElement)
        {
            // Get next trackpoint
            point   =log.get(element);

            dateTime=point.getDateTime();
            
            // If it is not the first point, calculate the time diff with
            // previous point
            if (prevPoint!=null)
            {
                prevDateTime=prevPoint.getDateTime();
                seconds=prevDateTime.numSecondsFrom(dateTime);
            }
            else
            {
                // If it is the 1st point, record start of track
                seconds=0;
            }
            // Check if this point is the start of a new track
            // The start is characterised by bitflags in the track point
            // Altenatively, a day change may be seen as end of track
            // However this does not make sense I think. Besides, the timestamp
            // is Zulu time. So depending where you use it, 0:00Z may be 
            // in the middle of the day
            
            // Check if this point is the start of a new segment.
            if ((point.isStartOfTrack()) || 
                ((segmentSeparation>0) && (seconds>segmentSeparation)))
            {
                // If any waypoint or trackpoint has been recorded,
                // the segment is finished
                if (waypointCount>0 || trackPointCount>0)
                {
                    endDateTime=prevPoint.getDateTime();

                    segment=new TrackSegment(trackSegmentNo,
                                             startElement,
                                             trackPointCount,
                                             waypointCount,
                                             startDateTime,
                                             endDateTime);
                    
                    track.appendTrackSegment(segment);
                    trackSegmentNo++;                    

                    // If a start of new track has been found,
                    // finish the track. This means: current point is 
                    // 1st point of next track.
                    if (point.isStartOfTrack())
                    {
                        
                        // Save the previous track
                        tracks.add(track);
                        
                        
                        // Create the next track
                        track=new Track();
                        
                        // prepare for next track
                        trackNo++;
                        trackSegmentNo=0;
                        
                    }

                }

                // Initialise for next segment
                startElement    =element;
                startDateTime   =point.getDateTime();                
                recordCount     =0;
                trackPointCount =0;
                waypointCount   =0;
            }

            
            

            if (!point.isWaypoint())
            {
                trackPointCount++;
            }
            else
            {
                waypointCount++;
            }
            recordCount++;

            prevPoint=point;
            prevDateTime=point.getDateTime();
            
            element++;
        }

        // Save remaining track
        if (waypointCount>0 || trackPointCount>0)
        {
            endDateTime=prevPoint.getDateTime();
            
            segment=new TrackSegment(trackSegmentNo,
                                     startElement,
                                     trackPointCount,
                                     waypointCount,
                                     startDateTime,
                                     endDateTime);
            track.appendTrackSegment(segment);
            trackSegmentNo++; 

        }
        
        // If any segment has been added to the final track, save it
        if (trackSegmentNo>0)
        {
            tracks.add(track);
        }

    }
*/    
    /**
     * This method parses the logs and adorns a heart rate value to the 
     * track log points.
     */
/*    
    private void parseHeartRates()
    {
        Iterator<TrackLogPoint>     trackLogIterator;
        Iterator<HeartRatePoint>    heartRateIterator;
        TrackLogPoint               trackPoint;
        HeartRatePoint              heartRatePoint;
        boolean                     exit;
        DateTime                    trackPointDateTime;
        DateTime                    heartRateDateTime;
        double                      sum;
        int                         count;
        
        // initial values
        heartRatePoint      =null;
        trackPoint          =null;
        
        // Obtain the iterators
        trackLogIterator    =this.log.iterator();
        heartRateIterator   =this.heartRateLog.iterator();

        // point of starting
        exit=false;
        
        // Get the 1st heart rate
        if (heartRateIterator.hasNext())
        {
            heartRatePoint=heartRateIterator.next();
        }
        else
        {
            heartRatePoint=null;
            exit=true;
        }
        
        // get the 1st track point that is not a waypoint
        if (trackLogIterator.hasNext())
        {
            trackPoint=trackLogIterator.next();
            // Skip waypoints, read until a track point read
            while (trackPoint.isWaypoint() && trackLogIterator.hasNext())
            {
                trackPoint=trackLogIterator.next();
            }
            
            // If the last point read isn't a track point, bail out
            if (trackPoint.isWaypoint())
            {
                trackPoint=null;
                exit=true;
            }
        }
        else
        {
            trackPoint=null;
            exit=true;
        }
        
        // initialise sum and count for average calculation
        sum=0.0;
        count=0;
        
        while (!exit)
        {
            // get the timestamps
            trackPointDateTime  =trackPoint.getDateTime();
            heartRateDateTime   =heartRatePoint.getDateTime();
            
            // If a change of track occurs: reset the average
            if (trackPoint.pointIsStartOfTrack)
            {
                sum=0.0;
                count=0;
            }
            
System.out.println("point "+trackPointDateTime.format("DD hh:mm:ss.fff") +
                   " heart "+heartRateDateTime.format("DD hh:mm:ss.fff")+ " sum "+sum+" cnt "+count);            
            

            if (heartRateDateTime.lteq(trackPointDateTime))
            {
                // heart rate values before or at the trackpoint: sum and count
                if (heartRatePoint.getHeartRateValue()>0)
                {
                    sum+=heartRatePoint.getHeartRateValue();
                    count++;
                }
                if (heartRateIterator.hasNext())
                {
                    heartRatePoint=heartRateIterator.next();
                }
                else
                {
System.out.println("No more trackpoints");                    
                    
                    heartRatePoint=null;
                    exit=true;
                }
            }
            else
            {
                // add average heartrate to trackpoint
                if (count>0)
                {
                    trackPoint.setHeartRate((int)(sum/count));
                    sum     =0.0;
                    count   =0;
                    
                }
                else
                {
                    trackPoint.setHeartRate(-1);
                }
System.out.println(">>> point "+trackPointDateTime.format("DD hh:mm:ss.fff") +
                   " heart "+heartRateDateTime.format("DD hh:mm:ss.fff")+ " value "+trackPoint.getHeartRate());            

                if (trackLogIterator.hasNext())
                {
                    trackPoint=trackLogIterator.next();
                    // skip waypoints
                    while (trackPoint.isWaypoint() && trackLogIterator.hasNext())
                    {
                        trackPoint=trackLogIterator.next();
                    }         
                    if (trackPoint.isWaypoint())
                    {
System.out.println("No more trackpoints 1");                    
                        trackPoint=null;
                        exit=true;
                    }                    
                }
                else
                {
System.out.println("No more trackpoints 2");                    
                    trackPoint=null;
                    exit=true;
                }
            } 
        }
        // If a heart rate average and a track point remaining, 
        // add the average to the trackpoint
        if (count>0 && trackPoint!=null)
        {
            trackPoint.setHeartRate((int)(sum/count));
System.out.println(">>>> point "+trackPoint.getDateTime().format("DD hh:mm:ss.fff") +
                   " value "+trackPoint.getHeartRate());            
        }

        // Set heart rate of any trackpoint left to -1
        while (trackLogIterator.hasNext())
        {
            trackPoint=trackLogIterator.next();
            // skip waypoints
            while (trackPoint.isWaypoint() && trackLogIterator.hasNext())
            {
                trackPoint=trackLogIterator.next();
            } 
            
            if (!trackPoint.isWaypoint())
            {
                trackPoint.setHeartRate(-1);
System.out.println(">>>>>> point "+trackPoint.getDateTime().format("DD hh:mm:ss.fff") +
                   " value "+trackPoint.getHeartRate());
            }
        }        

    }
*/    
    
    /**
     * This method finds the segment in the heart rate log given a start time
     * and an end time. The start index of the segment (heartRateLogSegmentStart) 
     * and the number of elements (heartRateLogSegmentNumber) are set.
     * @param start Start time
     * @param end End time
     */
/*    
    private void getHeartRateSegment(DateTime start, DateTime end)
    {
        HeartRatePoint  point;
        DateTime        dateTime;
        int             element;
        int             logSize;
        boolean         startFound;
        boolean         endFound;
        
        // Default values: not found
        this.heartRateLogSegmentStart=-1;
        this.heartRateLogSegmentNumber=0;
        
        endFound=false;
        startFound=false;
        element=0;
        logSize=this.heartRateLog.size();
        while ((element<logSize) && !endFound)
        {
            point   =this.heartRateLog.get(element);
            
            dateTime=point.getDateTime();
            
            if (!startFound)
            {
                if ((dateTime.gteq(start)) && (dateTime.lt(end)))
                {
                    this.heartRateLogSegmentStart=element;
                    startFound=true;
                }
            }
            else
            {
                if (dateTime.gt(end))
                {
                    this.heartRateLogSegmentNumber=element-this.heartRateLogSegmentStart;
                    endFound=true;
                }
            }            
            element++;
        }

        if (startFound && !endFound)
        {
            this.heartRateLogSegmentNumber=element-this.heartRateLogSegmentStart;
        }        
        
    }
*/    
    
    /**
     * Prints an overview of the segments to DebugLogger. Make sure debugging 
     * is on
     */
    void dumpTracks()
    {
        Iterator            iterator;
        Track               track;
        TrackSegment        segment;

        iterator=tracks.iterator();

        while (iterator.hasNext())
        {
            track=(Track)iterator.next();
            DebugLogger.info("Track: "+track.getTrackId());

            track.dumpSegments();
        }
    }

    /**
     * This method returns the number of track segments
     * @return The number of segments
     */
    public int getNumberOfTracks()
    {
        return tracks.size();
    }

    /**
     * This method returns the number of segments in given track
     * @param trackNumber The number of the track
     * @return The number of segments
     */
    public int getNumberOfTrackSegments(int trackNumber)
    {
        int     number;
        Track   track;

        if (trackNumber>=0 && trackNumber<tracks.size())
        {
            track=(Track)tracks.get(trackNumber);
            number=track.getNumberOfSegments();
        }
        else
        {
            number=0;
        }

        return number;
    }

    
    /**
     * This method returns all track points or all waypoints of given track
     * @param trackNumber The track
     * @param waypoint True for waypoints, false for trackpoints
     * @return Vector containing all track points
     */
/*    
    private ArrayList<TrackLogPoint> getLogPoints(int trackNumber, int segmentNumber, boolean waypoint)
    {
        ArrayList<TrackLogPoint>    points;
        TrackLogPoint               point;
        int                         records;
        int                         start;
        int                         i;
        Track                       track;
        TrackSegment                segment;
        
        points=new ArrayList<TrackLogPoint>();
        
        if (trackNumber>=0 && trackNumber<tracks.size())
        {
            track=(Track)tracks.get(trackNumber);

            segment=track.getSegment(segmentNumber);

            if (segment!=null)
            {
                records=segment.numberOfRecords;
                start=segment.startElement;

                i=0;
                while (i<records)
                {
                    point=log.get(start+i);
                    if (point.isWaypoint()==waypoint)
                    {
                        points.add(point);
                    }
                    i++;
                }
            }
        }
        
        return points;
    }
*/


    /**
     * This method returns all track points of given segment
     * @param trackNumber The Track
     * @param segmentNumber The segment
     * @return Vector containing all track points
     */
    public ArrayList<TrackLogPoint> getTrackPoints(int trackNumber, int segmentNumber)
    {
        return tracks.get(trackNumber).getSegment(segmentNumber).getTrackPoints();
    }


    /**
     * This method returns all track points of given track
     * @param trackNumber The Track
     * @return Vector containing all track points
     */
/*    
    public ArrayList<TrackLogPoint> getTrackPoints(int trackNumber)
    {
        int segment;
        int maxSegment;
        ArrayList<TrackLogPoint> theList;
        
        theList=new ArrayList<TrackLogPoint>();
        
        maxSegment=this.getNumberOfTrackSegments(trackNumber);
        segment=0;
        while (segment<maxSegment)
        {
            theList.addAll(tracks.get(trackNumber).getSegment(segment).getTrackPoints());
            segment++;
        }
        
        return theList;
    }
*/
    
    /**
     * This method returns all way points of given segment
     * @param segmentNumber
     * @return
     */
    public ArrayList<TrackLogPoint> getWayPoints(int trackNumber, int segmentNumber)
    {
        return tracks.get(trackNumber).getSegment(segmentNumber).getWayPoints();
    }

    /**
     * This method returns the start time of the indicated track.
     * @param trackNumber The ID of the track
     * @return The time in UTC of the start of the track
     */
    public DateTime getTrackStartTime(int trackNumber)
    {
        return tracks.get(trackNumber).getStartTime();
    }
    
    
    /**
     * This method return that part of the heart rate log that coincides 
     * with the track.
     * @param trackNumber Number of the track
     * @return Array list with heart rate values or null if no heart rate 
     *         available.
     */
    public ArrayList<HeartRatePoint> getTrackHeartRateLog(int trackNumber)
    {
        return tracks.get(trackNumber).getHeartRateValues();
    }
    
    /**
     * This method return that part of the heart rate log that coincides 
     * with the segment of indicated track.
     * @param trackNumber   Number of the track
     * @param segmentNumber Number of the segment
     * @return Array list with heart rate values or null if no heart rate 
     *         available.
     */
    public ArrayList<HeartRatePoint> getTrackSegmentHeartRateLog(int trackNumber, int segmentNumber)
    {
        return tracks.get(trackNumber).getSegment(segmentNumber).getHeartRatePoints();
    }
    
    
    
    /**
     * This method calculates the average heart rate for a track
     * @param trackNumber The number of the track
     * @return The average heart rate or -1 if no heart rate readings
     */
    public double getAverageHeartRate(int trackNumber)
    {
        return tracks.get(trackNumber).getAverageHeartRate();
    }
    

    /**
     * This method returns a descriptive String describing the indicated track
     * @param trackNumber Number of the track to describe
     * @return Description.
     */
    public String getTrackDescription(int trackNumber)
    {
        String  description;
        Track   track;

        if (trackNumber>=0 && trackNumber<tracks.size())
        {
            track=tracks.get(trackNumber);

            description=track.getDescription();


        }
        else
        {
            description="Non existing track";
        }

        return description;
    }




    /**
     * This method sets the device type of the device of which
     * the this TrackLog contains the track.
     * @param deviceType The device type as defined by Device class.
     *                   (e.g. DEVICETYPE_GT800)
     */
    public void setDeviceType(Device.DeviceType deviceType)
    {
        this.deviceType=deviceType;
    }

    /**
     * This method returns the type of the device that logged the track.
     * @return The device type as defined in the Device class.
     */
    public Device.DeviceType getDeviceType()
    {
        return this.deviceType;
    }
    
    /**
     * This method return indicated track
     * @param trackId Id of the track to retrieve
     * @return The indcated track
     */
    public Track getTrack(int trackId)
    {
        return this.tracks.get(trackId);
    }
}
