/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import java.util.TimeZone;


import java.util.ArrayList;
import java.util.Iterator;
import hirondelle.date4j.DateTime;

/**
 * This class represents a logged track. The track consists of segments.
 * A track starts when the device is logged on, it stop when the device is
 * switched off. Another start of track condition is when the device is reset
 * or the counters are reset.
 * A new segment is started if there is a separation between two 
 * adjacent track points that exceed the setting 'segmentSeparationLimit'.
 * If this setting is set to 0, the track will contain just one segment.
 * @author Jorgen
 */
public class Track
{
    private static int              nextTrackId=0;
    private static TrackLogPoint    previousTrackPoint=null;

    private int                     trackId;
    private ArrayList<TrackSegment> segments;
    
    private TrackSegment            currentSegment;
    
    private int                     segmentSeparation;
    
    /**
     * Constructor. Creates the vector with segments
     */
    public Track()
    {
        trackId             =nextTrackId;
        nextTrackId++;
        segments            =new ArrayList();
        
        // Reset the segment housekeeping
        TrackSegment.reset();
        
        // Creates the 1st segment under construction.
        currentSegment      =new TrackSegment();
        
        segmentSeparation=Settings.getInstance().getSegmentSeparationLimit();
        
    }

    /**
     * Resets the track. Makes it suitable for new download. THIS METHOD 
     * MUST BE CALLED PRIOR TO PARSING THE DATA AND CREATION OF TRACKS
     */
    public static void reset()
    {
        // Sets the ID housekeeping 
        nextTrackId         =0;
        previousTrackPoint  =null;
    }
    
    /**
     * Finish the track, i.e. add the pending segment. THIS METHOD MUST BE 
     * CALLED AFTER ALL DATA HAS BEEN PARSED. IF NOT DATA WILL BE LOST
     */
    public void finish()
    {
        if (!currentSegment.isEmptySegment())
        {
            // finish the segment
            currentSegment.finish();
            
            // add it to the list
            segments.add(currentSegment);
        }
    }
    
    /**
     * This method appends a trackpoint to the track.
     * @param point The trackpoint to add.
     */
    public void appendTrackpoint(TrackLogPoint point)
    {
        DateTime    dateTime;
        DateTime    prevDateTime;
        long        seconds;
        
        dateTime=point.getDateTime();

        // If it is not the first point, calculate the time diff with
        // previous point
        if (previousTrackPoint!=null)
        {
            prevDateTime=previousTrackPoint.getDateTime();
            seconds=prevDateTime.numSecondsFrom(dateTime);
        }
        else
        {
            // If it is the 1st point, record start of track
            seconds=0;
        }
        // Check if this point is the start of a new segment
        if ((segmentSeparation>0) && (seconds>segmentSeparation))
        {
            // If there is any data in current segment under construction, 
            // add it to the track and create a new current segment under 
            // construction. If no data, just keep it open
            if (!currentSegment.isEmptySegment())
            {
                // finish the segment
                this.currentSegment.finish();
                
                // add it to the list
                this.segments.add(currentSegment);
                
                // create the new segment-under-construction
                currentSegment=new TrackSegment();
            }
        }
        
        // Add the track point to the segment under construction.
        currentSegment.appendTrackpoint(point);
        
        // Store the point for next round
        previousTrackPoint=point;
    }

    /**
     * This method appends a waypoint to the segment
     * @param point The waypoint to append
     */
    public void appendWaypoint(TrackLogPoint point)
    {
        currentSegment.appendWaypoint(point);
    }
    
    /**
     * This method appends a list of heart rate values
     * @param point The heart rate values list to append
     */
    public void appendHeartRates(ArrayList<HeartRatePoint> points)
    {
        currentSegment.appendHeartRatePoints(points);
    }
    
    
    
    
    /**
     * Returns the start time of the track as UTC time.
     * This is the start time of the 1st segment in the track.
     * @return The start datetime
     */
    public DateTime getStartTime()
    {
        DateTime returnTime;

        if (segments.size()>0)
        {
            returnTime=segments.get(0).getStartTime();
        }
        else
        {
            returnTime=null;
        }
        return returnTime;
    }


    /**
     * Returns the start time of the track as local time.
     * This is the start time of the 1st segment in the track.
     * @return The start datetime
     */
    public DateTime getLocalStartTime()
    {
        DateTime gmtTime;
        DateTime localTime;
        TimeZone gmtTimeZone;
        TimeZone localTimeZone;

        gmtTimeZone=TimeZone.getTimeZone("GMT");
        localTimeZone=TimeZone.getDefault();

        gmtTime=getStartTime();
        if (gmtTime!=null)
        {
            localTime=gmtTime.changeTimeZone(gmtTimeZone, localTimeZone);
        }
        else
        {
            localTime=null;
        }
        return localTime;

    }

    /**
     * Returns the end time of the track as UTC time.
     * This is the end time of the last segment in the track.
     * @return The end datetime
     */
    public DateTime getEndTime()
    {
        DateTime returnTime;

        if (segments.size()>0)
        {
            returnTime=segments.get(segments.size()-1).getEndTime();
        }
        else
        {
            returnTime=null;
        }
        return returnTime;
    }

    /**
     * Returns the end time of the track as local time.
     * This is the end time of the last segment in the track.
     * @return The start datetime
     */
    public DateTime getLocalEndTime()
    {
        DateTime gmtTime;
        DateTime localTime;
        TimeZone gmtTimeZone;
        TimeZone localTimeZone;

        gmtTimeZone=TimeZone.getTimeZone("GMT");
        localTimeZone=TimeZone.getDefault();

        gmtTime=getEndTime();
        if (gmtTime!=null)
        {
            localTime=gmtTime.changeTimeZone(gmtTimeZone, localTimeZone);
        }
        else
        {
            localTime=null;
        }
        return localTime;

    }
    

    /**
     * This method returns the ID of the track
     * @return The id
     */
    public int getTrackId()
    {
        return trackId;
    }

    /**
     * This method prints information on the segments that make up the track.
     * It is written to the debuglogger. Make sure it is on.
     */
    public void dumpSegments()
    {
        Iterator            iterator;
        TrackSegment        segment;

        iterator=segments.iterator();

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();

            DebugLogger.info("Segment "+segment.getSegmentId()+
                               " "+segment.getStartTime().format("YYYY-MM-DD hh:mm")+
                               " "+segment.getEndTime().format("YYYY-MM-DD hh:mm")+
                               " track "+segment.getNumberOfTrackpoints()+
                               " wp "+segment.getNumberOfWaypoints()+
                               " heart "+segment.getNumberOfHeartRatePoints());
        }
    }

    /**
     * This method returns the list of track segments.
     * @return The ArrayList with segments
     */
    public ArrayList<TrackSegment> getSegments()
    {
        return this.segments;
    }
    
    /**
     * This method returns the segments that make up the track.
     * @return
     */
    public TrackSegment getSegment(int segmentNumber)
    {
        TrackSegment segment;

        if (segmentNumber>=0 && segmentNumber<segments.size())
        {
            segment=segments.get(segmentNumber);
        }
        else
        {
            segment=null;
        }
        return segment;
    }

    /**
     * This method returns the number of segments in the track
     * @return The number of segments
     */
    public int getNumberOfSegments()
    {
        return segments.size();
    }

    /**
     * This method returns the total number of tackpoints (sum of segments)
     * (Trackpoints are points logged by the device at regular intervals)
     * @return The number
     */
    public int getNumberOfTrackPoints()
    {
        int                 number;
        Iterator            iterator;
        TrackSegment        segment;

        iterator=segments.iterator();
        number=0;

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();

            number+=segment.getNumberOfTrackpoints();
        }
       return number;
    }

   /**
    * This method returns the total number of waypoints (sum of segments)
    * (Waypoints are points logged by the user durig the trip)
    * @return The number
    */
    public int getNumberOfWaypoints()
    {
        int                 number;
        Iterator            iterator;
        TrackSegment        segment;

        iterator=segments.iterator();
        number=0;

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();

            number+=segment.getNumberOfWaypoints();
        }
       return number;
    }

    /**
     * This method returns the total number of heart rate values in this 
     * track.
     * @return The number of heart rate points 
     */
    public int getNumberOfHeartRatePoints()
    {
        int                 number;
        Iterator            iterator;
        TrackSegment        segment;

        iterator=segments.iterator();
        number=0;

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();

            number+=segment.getNumberOfHeartRatePoints();
        }
       return number;        
    }
    
    
    /**
     * This method returns all heart rate values in this track in one list.
     * @return ArrayList with all heart rate values
     */
    public ArrayList<HeartRatePoint> getHeartRateValues()
    {
        ArrayList<HeartRatePoint>   rates;
        Iterator                    iterator;
        TrackSegment                segment;

        rates=new ArrayList<HeartRatePoint>();
        
        iterator=segments.iterator();
        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();
            rates.addAll(segment.getHeartRatePoints());    
        }
        return rates;
    }
    
    
    /**
     * This method calculates the average heart rate for the track
     * @return The average heart rate or -1 if no heart rate readings
     */
    public double getAverageHeartRate()
    {
        ArrayList<HeartRatePoint>   rates;
        Iterator<HeartRatePoint>    iterator;
        HeartRatePoint              point;
        double                      averageHeartRate;
        int                         count;
        int                         heartRate;
        
        rates=getHeartRateValues();
        
        averageHeartRate        =0.0;
        count                   =0;
        iterator=rates.iterator();
        
        while (iterator.hasNext())
        {
            point=iterator.next();
            heartRate=point.getHeartRateValue();
            if (heartRate>0)
            {
                count++;
                averageHeartRate+=heartRate;
            }        
        }
        
        if (count>0)
        {
            averageHeartRate/=count;
        }
        else
        {
            averageHeartRate=-1.0;
        }
        return averageHeartRate;
        
    }

    /**
     * This method calculates the maximum heart rate for the track
     * @return The average heart rate or -1 if no heart rate readings
     */
    public int getMaxHeartRate()
    {
        ArrayList<HeartRatePoint>   rates;
        Iterator<HeartRatePoint>    iterator;
        HeartRatePoint              point;
        int                         max;
        int                         heartRate;
        
        rates=getHeartRateValues();
        
        max                     =-1;
        iterator=rates.iterator();
        
        while (iterator.hasNext())
        {
            point=iterator.next();
            heartRate=point.getHeartRateValue();
            if ((heartRate>0) && (heartRate>max))
            {
                max=heartRate;
            }        
        }
        return max;
    }
    
    
    /**
     * Calculate the calories burned during the track, based on heart rate.
     * @return The number of kCal, or -1 if not possible to estimate (no heart rate)
     */
    public double getCalories()
    {
        double              calories;
        Iterator            iterator;
        TrackSegment        segment;

        iterator    =segments.iterator();
        calories    =0.0;

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();

            calories+=segment.getSegmentCalories();
        }
       return calories;        
        
/*        
        Settings    settings;
        double      calories;
        double      age;
        double      weight;
        String      gender;
        double      averageHeartRate;
        double      duration;
        
        calories        =0.0;
        
        settings        =Settings.getInstance();
        age             =settings.getProfileAge();
        weight          =settings.getProfileWeight();
        gender          =settings.getProfileGender();
        averageHeartRate=this.getAverageHeartRate();
        duration        =this.getTrackDuration();
        
        
        if (averageHeartRate>0)
        {
            // age in year, weigth in kg, heart rate in bpm, duration in minutes -> result kCal
            if (gender.equals("male"))
            {
                calories = ( (age * 0.2017) + (weight * 0.1988)+ (averageHeartRate * 0.6309) - 55.0969) * (duration/60) / 4.184;
            }
            else
            {
                calories = ( (age * 0.074) + (weight * 0.1263) + (averageHeartRate * 0.4472) - 20.4022) * (duration/60) / 4.184;       
            }
        }
        else
        {
            calories=0.0;
        }
         

        return calories;         
*/
    }
    
    
    
    /**
     * This method returns whether the track is empty, i.e. contains no data.
     * @return True if the track contains no data, false if there is data
     */
    public boolean isEmptyTrack()
    {
       boolean isEmpty=true;
       
       // Check if there are segments (a segment added always contains data)
       if (segments.size()>0)
       {
           isEmpty=false;
       }
       else
       {
           // If no segments added, check if the segment under construction
           // contains data
           if (!currentSegment.isEmptySegment())
           {
               isEmpty=false;
           }
       }
       return isEmpty;
    }
    
    
    /**
     * Returns a description of the track
     * @return The description as a String
     */
    public String getDescription()
    {
        String          description;

        description="Track "+String.format("%4d", getTrackId())+
                    " "+getLocalStartTime().format("YYYY-MM-DD hh:mm")+
                    " - "+getLocalEndTime().format("hh:mm")+
                    " segments: "+ String.format("%4d", getNumberOfSegments())+
                    " trackpnts: "+String.format("%6d", getNumberOfTrackPoints())+
                    " waypnts: "+String.format("%6d", getNumberOfWaypoints())+
                    " heartrate "+String.format("%6d", getNumberOfHeartRatePoints());



        return description;
    }


    /**
     * This method returns the distance traveled by this track. It
     * is the sum of distances traveled by  the segments
     * @return The distance in meter
     */
    public double getTrackDistance()
    {
        double              distance;
        Iterator            iterator;
        TrackSegment        segment;

        iterator=segments.iterator();
        distance=0.0;

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();

            distance+=segment.getSegmentDistance();
        }
       return distance;
    }
    
    /**
     * This method returns the maximum speed logged during the track.
     * @return The maximum speed value in m/s.
     */
    public double getTrackMaxSpeed()
    {
        double              speedMax;
        double              speed;
        Iterator            iterator;
        TrackSegment        segment;

        iterator=segments.iterator();
        speedMax=0.0;

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();
            speed=segment.getSegmentMaxSpeed();
            if (speed>speedMax)
            {
                speedMax=speed;
            }

        }
       return speedMax;
    }
    
    /**
     * This method returns the duration of the track. This is the total
     * duration of the segments, excluding the periods in between.
     * @return Duration in seconds
     */
    public long getTrackDuration()
    {
        long                duration;
        Iterator            iterator;
        TrackSegment        segment;

        iterator=segments.iterator();
        duration=0;

        while (iterator.hasNext())
        {
            segment=(TrackSegment)iterator.next();

            duration+=segment.getSegmentDuration();
        }
       return duration;
    }
    
    /**
     * This method returns the total duration of the track. This is the total
     * duration of the segments, including the periods in between.
     * @return Duration in seconds
     */
    public long getTotalTrackDuration()
    {
        long                duration;
        Iterator            iterator;
        TrackSegment        segment;
        int                 size;
        DateTime            start;
        DateTime            end;

        duration=0;

        size=segments.size();
        
        if (size>0)
        {
            start=segments.get(0).getStartTime();
            end=segments.get(size-1).getEndTime();
            duration=start.numSecondsFrom(end);
        }
        return duration;
    }
    

    
}
