/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import java.util.Properties;
import java.util.TimeZone;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import hirondelle.date4j.DateTime;
/**
 *
 * @author UserXP
 */
public class Settings
{
    private static  Settings theInstance=null;

    private String  propertyFileName="u-gotme.properties";

    
    private boolean     debugging;
    private int         debugLevel;
    private String      defaultComport;
    private String      defaultGpxPath;
    private String      defaultLogPath;
    private String      outputFileType;
    private String      gpxVersion;
    private String      mapType;
    private String      operationMode;
    private String      simulationPath;
    private int         segmentSeparationSeconds;
    private String      profileGender;
    private double      profileWeight;
    private double      profileLength;
    private DateTime    profileDateOfBirth;
    private double      metabolicEquivalent;
    private boolean     trackCaching;
    private String      comportLib;
    private String      fontsize;
    
    private Settings()
    {
        readSettings();
    }

    private void readSettings()
    {
        Properties      properties;
        String          setting;

        defaultComport=null;
        gpxVersion=null;

        // Read properties file.
        properties = new Properties();
        try
        {
            properties.load(new FileInputStream(propertyFileName));

            setting=properties.getProperty("debugLevel");
            this.setDebugLevel(setting);
            // As a side effect, set the DebugLogger accordingly, so
            // debug logging starts right away
            DebugLogger.setDebugLevel(debugLevel);


            setting=properties.getProperty("comport");
            this.setComport(setting);

            setting=properties.getProperty("outputFileType");
            this.setOutputFileType(setting);

            setting=properties.getProperty("gpxVersion");
            this.setGpxVersion(setting);

            setting=properties.getProperty("gpxPath");
            this.setGpxPath(setting);

            setting=properties.getProperty("logPath");
            this.setLogPath(setting);
            
            setting=properties.getProperty("mapType");
            this.setMapType(setting);
            
            setting=properties.getProperty("segmentSeparation");
            this.setSegmentSeparationLimit(setting);

            setting=properties.getProperty("operationMode");
            this.setOperationMode(setting);
            
            setting=properties.getProperty("simulationPath");
            this.setSimulationPath(setting);
            
            setting=properties.getProperty("debugging");
            this.setDebugging(setting);
            
            setting=properties.getProperty("profileGender");
            this.setProfileGender(setting);
            
            setting=properties.getProperty("profileWeight");
            this.setProfileWeight(setting);
            
            setting=properties.getProperty("profileLength");
            this.setProfileLength(setting);
            
            setting=properties.getProperty("profileDateOfBirth");
            this.setProfileDateOfBirth(setting);
            
            setting=properties.getProperty("metabolicEquivalent");
            this.setMetabolicEquivalent(setting);
            
            setting=properties.getProperty("trackCaching");
            this.setTrackCaching(setting);

            setting=properties.getProperty("comportLib");
            this.setComportLib(setting);
            
            setting=properties.getProperty("routepointFontsize");
            this.setRoutepointFontsize(setting);
            
            
            DebugLogger.info("Settings read");
            dumpSettings();
        }
        catch (IOException e)
        {
            DebugLogger.error("Error reading settings from "+propertyFileName);
            debugLevel              =DebugLogger.DEBUGLEVEL_ERROR;
            defaultComport          ="";
            defaultGpxPath          ="";
            defaultLogPath          ="";
            outputFileType          ="GPX";
            gpxVersion              ="1.1";
            mapType                 ="roadmap";
            segmentSeparationSeconds=0;
            operationMode           ="normal";
            simulationPath          ="";
            profileWeight           =75.0;
            profileGender           ="male";
            profileLength           =175.0;
            profileDateOfBirth      =new DateTime("2000-01-01");
            metabolicEquivalent     =2.0;
            trackCaching            =false;
            comportLib              ="rxtx";
            fontsize                ="medium";
        }

    }

    /**
     * Writes current settings to the properties file
     */
    public void writeSettings()
    {
        Properties      properties;
        // Read properties file.
        properties = new Properties();
        try
        {
            properties.setProperty("debugging"          , Boolean.toString(debugging));
            properties.setProperty("debugLevel"         , DebugLogger.debugLevelToString(debugLevel));
            properties.setProperty("trackCaching"       , Boolean.toString(trackCaching));
            
            properties.setProperty("gpxPath"            , defaultGpxPath);
            properties.setProperty("logPath"            , defaultLogPath);
            properties.setProperty("comport"            , defaultComport);
            properties.setProperty("outputFileType"     , outputFileType);
            properties.setProperty("gpxVersion"         , gpxVersion);
            properties.setProperty("mapType"            , mapType);
            properties.setProperty("segmentSeparation"  , Integer.toString(segmentSeparationSeconds));
            properties.setProperty("operationMode"      , operationMode);
            properties.setProperty("simulationPath"     , simulationPath);    
            properties.setProperty("profileWeight"      , Double.toString(profileWeight));
            properties.setProperty("profileLength"      , Double.toString(profileLength));
            properties.setProperty("profileGender"      , profileGender);
            properties.setProperty("profileDateOfBirth" , profileDateOfBirth.format("YYYY-MM-DD"));
            properties.setProperty("metabolicEquivalent", Double.toString(metabolicEquivalent));
            properties.setProperty("comportLib"         , comportLib);
            properties.setProperty("routepointFontsize" , fontsize);

            properties.store(new FileOutputStream(propertyFileName), "");


            DebugLogger.info("Settings written");
            dumpSettings();

        }
        catch (IOException e)
        {
            DebugLogger.error("Error writing properties file "+propertyFileName);
        }
    }

    /**
     * Output settings to the debug output
     */
    private void dumpSettings()
    {
        DebugLogger.info("SETTINGS FROM PROPERTY FILE");
        DebugLogger.info("Setting default comport  (comport)            : "+defaultComport);
        DebugLogger.info("Setting default GPX path (gpxPath)            : "+defaultGpxPath);
        DebugLogger.info("Setting default log path (logPath)            : "+defaultLogPath);
        DebugLogger.info("Output file preference   (outputFileType)     : "+outputFileType);
        DebugLogger.info("Setting GPX Version      (gpxVersion)         : "+gpxVersion);
        DebugLogger.info("Debugging                (debugging)          : "+debugging);
        DebugLogger.info("Track caching to file    (trackCaching)       : "+trackCaching);
        DebugLogger.info("Setting debug level      (debugLevel)         : "+DebugLogger.debugLevelToString(debugLevel));
        DebugLogger.info("Setting map type         (mapType)            : "+mapType);
        DebugLogger.info("Segment separation (sec) (segmentSeparation)  : "+segmentSeparationSeconds);
        DebugLogger.info("Mode of operation        (operationMode)      : "+operationMode);
        DebugLogger.info("Simulation path          (simulationPath)     : "+simulationPath);
        DebugLogger.info("Profile date-of-birth    (profileDateOfBirth) : "+profileDateOfBirth.format("YYYY-MM-DD")+
                                                                       " ("+this.getProfileAge()+")");
        DebugLogger.info("Profile gender           (profileGender)      : "+profileGender);
        DebugLogger.info("Profile weight           (profileWeight)      : "+profileWeight);
        DebugLogger.info("Profile length           (profileLength)      : "+profileLength);
        DebugLogger.info("Metabolic Equivalent     (metabolicEquivalent): "+metabolicEquivalent);
        DebugLogger.info("Comport library          (comportLib)         : "+comportLib);
        DebugLogger.info("Route point fontsize     (routepointFontsize) : "+fontsize);
        
        
    }


    /**
     * Returns the property file name
     * @return The filename
     */
    public String getPropertyFileName()
    {
        return propertyFileName;
    }

    /**
     * Returns the one and only singleton instance of this class
     * @return The instance
     */
    public static Settings getInstance()
    {
        if (theInstance==null)
        {
            theInstance=new Settings();
        }
        return theInstance;
    }

    /**
     * Gets the debugging level setting
     * @return The debug level conform the definition in DebugLogger.
     */
    public int getDebugLevel()
    {
        return debugLevel;
    }

    /**
     * Sets the debugging setting
     * @param debugLevel The debug level conform the definition in DebugLogger.
     */
    public void setDebugLevel(int debugLevel)
    {
        this.debugLevel=debugLevel;
    }
    
    /**
     * Sets the debugging setting
     * @param debugLevel 'off', 'debug', 'info' or 'error'
     */
    public void setDebugLevel(String debugLevel)
    {
        // Default debug level: only show the errors
        this.debugLevel=DebugLogger.DEBUGLEVEL_ERROR;
        
        if (debugLevel!=null)
        {
            if (debugLevel.toLowerCase().equals("off"))
            {
                this.debugLevel=DebugLogger.DEBUGLEVEL_OFF;
            }
            else if (debugLevel.toLowerCase().equals("debug"))
            {
                this.debugLevel=DebugLogger.DEBUGLEVEL_DEBUG;
            }
            else if (debugLevel.toLowerCase().equals("info"))
            {
                this.debugLevel=DebugLogger.DEBUGLEVEL_INFO;
            }
            else if (debugLevel.toLowerCase().equals("error"))
            {
                this.debugLevel=DebugLogger.DEBUGLEVEL_ERROR;
            }
        }
    }

    /**
     * This method returns the default path for saving GPX files.
     * @return The path or "" if the path is not defined
     */
    public String getGpxPath()
    {
        return defaultGpxPath;
    }

    /**
     * This method sets the default path for saving GPX files. If null is passed
     * the default path will become ""
     * @param newPath The new path
     */
    public void setGpxPath(String newPath)
    {
        if (newPath!=null)
        {
            this.defaultGpxPath=newPath;
        }
        else
        {
            this.defaultGpxPath="";
        }
    }

    /**
     * This method returns the default path for saving log files.
     * @return The path or "" if the path is not defined
     */
    public String getLogPath()
    {
        return defaultLogPath;
    }

    /**
     * This method sets the default path for saving log files. If null is passed
     * the default path will become ""
     * @param newPath The new path
     */
    public void setLogPath(String newPath)
    {
        if (newPath!=null)
        {
            this.defaultLogPath=newPath;
        }
        else
        {
            this.defaultLogPath="";
        }
        
        // Make sure the log path ends with a slash, if it is not empty.
        // A not defined log path means the log is stored in the startup
        // directory
        if (!this.defaultLogPath.equals(""))
        {
            if (!(this.defaultLogPath.endsWith("/") || 
                  this.defaultLogPath.endsWith("\\")))
            {
                this.defaultLogPath+="/";
            }
        }
    }

    /**
     * This method sets the output file type.
     * @param fileType The type: null, "GPX", "CSV" or "TCX"
     */
    public void setOutputFileType(String fileType)
    {
        if (fileType!=null)
        {
            if (fileType.toUpperCase().equals("TCX"))
            {
                this.outputFileType="TCX";
            }
            else if (fileType.toUpperCase().equals("CSV"))
            {
                this.outputFileType="CSV";
            }
            else
            {
                this.outputFileType="GPX";
            }
               
        }
        else
        {
            this.outputFileType="GPX";
        }
    }
    
   /**
     * This method returns the output file type
     * @return "GPX", "TCX" or "CSV"
     */
    public String getOutputFileType()
    {
        return this.outputFileType;
    }
    
    
    /**
     * Returns the GPX version to be used
     * @return "1.0" or "1.1"
     */
    public String getGpxVersion()
    {
        return gpxVersion;
    }

    /**
     * This method sets the GPX version. If null is passed, the default
     * value will be 1.1
     * @param newVersion "1.0" or "1.1"
     */
    public void setGpxVersion(String newVersion)
    {
        // the default
        this.gpxVersion="1.1";

        if (newVersion!=null)
        {
            if (newVersion.equals("1.0"))
            {
                this.gpxVersion="1.0";
            }
        }

    }

    /**
     * This method returns the default comport.
     * @return The default comport (e.g. "COM3") or "" if not defined.
     */
    public String getComport()
    {
        return defaultComport;
    }

    /**
     * This method sets the default comport. When null is passed, the
     * new comport will be ""
     * @param newComport The new value or null if not defined.
     */
    public void setComport(String newComport)
    {
        defaultComport="";
        if (newComport!=null)
        {
            defaultComport=newComport;
        }
    }

    /**
     * This method returns the map type to be used for plotting tracks, 
     * waypoints and routes on static Google maps
     * @return The map type ('roadmap', 'satellite', 'terrain' or 'hybrid')
     */
    public String getMapType()
    {
       
        return mapType;
    }
    
    /**
     * This method sets the map type
     * @param newMapType The new map type ('roadmap', 'satellite', 'terrain' or 'hybrid').
     */
    public void setMapType(String newMapType)
    {
        // The default type
        mapType="roadmap";
        
        if (newMapType!=null)
        {
            mapType=newMapType;
        }
    }

    /**
     * This method returns the minimum number of seconds that should
     * be between two track points before the next trackpoint is regarded
     * as start of a new segment. Or 0 if all tracks should consist of one
     * segment.
     * @return The number of seconds
     */
    public int getSegmentSeparationLimit()
    {
        return this.segmentSeparationSeconds;
    }

    /**
     * This method sets a new value to the number of seconds that should
     * be between two track points before the next trackpoint is regarded
     * as start of a new segment. Set to 0 if all tracks should consist of one
     * segment, i.e. all trackpoints are added in one segment.
     * @param newLimit The new separation limit value in seconds
     */
    public void setSegmentSeparationLimit(int newLimit)
    {
        this.segmentSeparationSeconds=newLimit;
    }

    /**
     * This method sets a new value to the number of seconds that should
     * be between two track points before the next trackpoint is regarded
     * as start of a new segment. Set to 0 if all tracks should consist of one
     * segment, i.e. all trackpoints are added in one segment.
     * @param newLimit The new separation limit value in seconds
     */
    public void setSegmentSeparationLimit(String newLimit)
    {
        if (newLimit!=null)
        {
            this.segmentSeparationSeconds=Integer.parseInt(newLimit);
        }
    }
    
    /**
     * This method sets the simulation value. It is the path where to find
     * simulation data. 
     * @param newSimulation New value for simulation path
     */
    private void setSimulationPath(String newSimulation)
    {
        simulationPath="";
        if (newSimulation!=null)
        {
            simulationPath=newSimulation;
        }
    }
    
    /**
     * This method returns the path where to find Device simulation data.
     * @return The path or null when the device is used in real mode.
     */
    public String getSimulationPath()
    {
        return simulationPath;
    }
    

    /**
     * This method sets the the system to debugging
     * @param newState "true" for debugging, "false" for normal operation
     */
    public void setDebugging(String newState)
    {
        debugging=false;
        if (newState!=null)
        {
            if (newState.equals("true"))
            {
                debugging=true;
            }
        }
    }
    
    /**
     * This method returns whether the system in in debugging state
     * @return True if in debugging, false if normal operation
     */
    public boolean getDebugging()
    {
        return debugging;
    }
    
    /**
     * This method sets the the system to debugging
     * @param newState "true" for debugging, "false" for normal operation
     */
    public void setTrackCaching(String newState)
    {
        trackCaching=false;
        if (newState!=null)
        {
            if (newState.equals("true"))
            {
                trackCaching=true;
            }
        }
    }
    
    /**
     * This method returns whether the system in in debugging state
     * @return True if in debugging, false if normal operation
     */
    public boolean getTrackCaching()
    {
        return trackCaching;
    }

    
    
    
   
    /**
     * This method returns the map type to be used for plotting tracks, 
     * waypoints and routes on static Google maps
     * @return The map type ('roadmap', 'satellite', 'terrain' or 'hybrid')
     */
    public String getOperationMode()
    {
       
        return operationMode;
    }
    
    /**
     * This method sets the operation mode
     * @param newOperationMode New mode of operation ('simulation' or 'normal')
     */
    public void setOperationMode(String newOperationMode)
    {
        // The default type
        operationMode="normal";
        
        if (newOperationMode!=null)
        {
            operationMode=newOperationMode;
        }
    }
    
    /**
     * This method indicates whether the software runs in simulation mode.
     * 
     * @return true for simulation mode, false for the real stuff
     */
    public boolean isSimulationMode()
    {
        return (operationMode.equals("simulation"));
    }
     
    /**
     * This method sets a new gender value
     * @param gender The gender value as string: "male" or "female"
     */
    public void setProfileGender(String gender)
    {
        if (gender!=null)
        {
            if (gender.toLowerCase().equals("female"))
            {
                this.profileGender="female";
            }
            else
            {
                this.profileGender="male";
            }
        }
        else
        {
            this.profileGender="male";
        }
    }
    
    /**
     * This method returns the gender as string.
     * @return "male" or "female"
     */
    public String getProfileGender()
    {
        return this.profileGender;
    }
    
    /**
     * This method sets the date-of-birth as string.
     * @param dob Date of birth: yyyy-mm-dd
     */
    public void setProfileDateOfBirth(String dob)
    {
        if (dob!=null)
        {
            this.profileDateOfBirth=new DateTime(dob);
        }
        else
        {
            this.profileDateOfBirth=new DateTime("2000-01-01");
        }
    }
    
    /**
     * This method returns the date-of-birth
     * @return The Date-of-birth as DateTime
     */
    public DateTime getProfileDateOfBirth()
    {
        return this.profileDateOfBirth;
    }
    
    /**
     * This method returns the current age based on the date-of-birth
     * @return The age in years.
     */
    public int getProfileAge()
    {
        DateTime    now;
        int         age;
        
        now=DateTime.now(TimeZone.getDefault());
        
        // year difference
        age=now.getYear()-this.profileDateOfBirth.getYear();
       
        // if birthday has not passed this year, age --

        if (
             (this.profileDateOfBirth.getMonth()>now.getMonth()) || 
             (
                (this.profileDateOfBirth.getMonth()==now.getMonth()) && 
                (this.profileDateOfBirth.getDay()>now.getDay())
             )
           )
        {
            age--;
        }
 
        return age;
    }
    
    /**
     * This method sets the profile weight encoded in a string
     * @param weight The weight in kg, like "85"
     */
    public void setProfileWeight(String weight)
    {
        if (weight!=null)
        {
            this.profileWeight=Double.valueOf(weight);
        }
        else
        {
            this.profileWeight=75.0;
        }
    }
    
    /**
     * This method sets the profile weight
     * @param newWeight The weight in kg.
     */
    public void setProfileWeight(double newWeight)
    {
        this.profileWeight=newWeight;
    }
    
    /**
     * This method returns the profile weight in kg
     * @return The weight in kg
     */
    public double getProfileWeight()
    {
        return this.profileWeight;
    }

   /**
     * This method sets the profile length encoded in a string
     * @param length The length in cm, like "185.0"
     */
    public void setProfileLength(String length)
    {
        if (length!=null)
        {
            this.profileLength=Double.valueOf(length);
        }
        else
        {
            this.profileLength=175.0;
        }
    }
    
    /**
     * This method sets the profile length
     * @param newLength The length in cm.
     */
    public void setProfileLength(double newLength)
    {
        this.profileLength=newLength;
    }
    
    /**
     * This method returns the profile legnth in cm
     * @return The legnth in cm
     */
    public double getProfileLength()
    {
        return this.profileLength;
    }

     
    /**
     * This method sets the metabolic rate used for calorie calculation
     * @param equivalent The new metabolic equivalent value
     */
    public void setMetabolicEquivalent(String equivalent)
    {
        if (equivalent!=null)
        {
            this.metabolicEquivalent=Double.valueOf(equivalent);
        }
        else
        {
            this.metabolicEquivalent=2.0;
        }
    }
    
    /**
     * This method returns the metabolic equivalent value
     * @return The metabolic equivalent value
     */
    public double getMetabolicEquivalent()
    {
        return this.metabolicEquivalent;
    }
    
    /**
     * This method sets the comport library to be used
     * @param lib The comport lib value as string: "rxtx" (default), "purejava" or "jssc"
     */
    public void setComportLib(String lib)
    {
        if (lib!=null)
        {
            if (lib.toLowerCase().equals("purejava"))
            {
                this.comportLib="purejava";
            }
            else if (lib.toLowerCase().equals("jssc"))
            {
                this.comportLib="jssc";
            }
            else
            {
                this.comportLib="rxtx";
            }
        }
        else
        {
            this.comportLib="rxtx";
        }
    }
    
    /**
     * This method returns the comport library to be used
     * @return The comport library
     */
    public String getComportLib()
    {
        return this.comportLib;
    }
    

    /**
     * This method sets the font size that is used for the labels in the route log
     * @param fontsize String describing the fontsize 'small', 'medium' or 'large' 
     */
    public void setRoutepointFontsize(String fontsize)
    {
        if (fontsize!=null)
        {
            if (fontsize.toLowerCase().equals("small"))
            {
                this.fontsize="small";
            }
            else if (fontsize.toLowerCase().equals("large"))
            {
                this.fontsize="large";
            }
            else
            {
                this.fontsize="medium";
            }
        }
        else
        {
            this.fontsize="medium";
        }
    }
    
    /**
     * This method returns the fontsize for the labels for the routepoint
     * @return The fontsize
     */
    public String getRoutepointFontsize()
    {
        return this.fontsize;
    }

}
