/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import java.awt.image.BufferedImage;
import java.awt.Graphics2D;
import java.awt.Font;

/**
 * This class represents a route waypoint
 * @author Jorgen
 */
public class RoutePoint extends Coordinate
{
    /** The size of the record representing a route point on the GT-800 */
    public static final int ROUTEPOINT_RECORDSIZE   =0x120;

    /** The size of the part of the record representing the bitmap */
    public static final int BITMAP_SIZE             =0x100;

    /** Bitmap width */
    public static final int BITMAP_WIDTH            =128;

    /** Bitmap height */
    public static final int BITMAP_HEIGHT           =16;
    
    /** X coordinate of the vertical separator */
    public static final int BITMAP_SEPARATORX       =29;

    /** X coordinate of the symbol (left side) */
    public static final int BITMAP_SYMBOLX          =31;

    /** X coordinate of the text field representing the name */
    public static final int BITMAP_NAMEX            =47;

    /** It is what it suggests it is */
    public static final int BITS_PER_BYTE           =8;


    /** Bitmap that is shown on top of the display, representing the name */
    private byte[]          bitmap;

    /** Name of the waypoint */
    private String          name;

    /** Name of the symbol associated with the route point */
    private String          symbol;


    /** Byte array representing the record in the format the Device expects it */
    private static byte[]   record=null;

    /** Image used for rendering the bitmap */
    private static BufferedImage image=null;


    /** The symbol repository instance */
    private static SymbolRepository symbolRepository=null;

    
    /**
     * Helper intialising the static variables
     */
    private void initStatics()
    {
        if (record==null)
        {
            record=new byte[ROUTEPOINT_RECORDSIZE];
        }

        if (image==null)
        {
            image=new BufferedImage(BITMAP_WIDTH, BITMAP_HEIGHT, BufferedImage.TYPE_BYTE_BINARY);
        }

        if (symbolRepository==null)
        {
            symbolRepository=SymbolRepository.getInstance();
        }

    }

    /**
     * This method creates the waypoint from the record data received from the
     * Device.
     * @param data The data. 0x120 bytes
     * @param routePointNumber number of the route point. It is used if 
     *        no name was found.
     */
    public RoutePoint(byte[] data, int routePointNumber)
    {
        int    i;
        int    symbolId;
        byte[] nameBytes;


        initStatics();

        // MEMORY MAP:
        // @trip:  |start|lat  |lon  |elev |not used 0x00 |bitmap   |
        // u-gotme:|start|lat  |lon  |elev |name | symbol |bitmap   |
        //         00    04    08    12    16    31       32        288 
        
        latitude =((data[ 4]&0xff))+
                  ((data[ 5]&0xff)<<8)+
                  ((data[ 6]&0xff)<<16)+
                  ((data[ 7])<<24);
        longitude=((data[ 8]&0xff))+
                  ((data[ 9]&0xff)<<8)+
                  ((data[10]&0xff)<<16)+
                  ((data[11])<<24);
        elevation=((data[12]&0xff))+
                  ((data[13]&0xff)<<8)+
                  ((data[14]&0xff)<<16)+
                  ((data[15])<<24);

        // create bitmap
        bitmap=new byte[RoutePoint.BITMAP_SIZE];

        // Copy the bitmap
        i=0;
        while (i<RoutePoint.BITMAP_SIZE)
        {
            bitmap[i]=data[32+i];
            i++;
        }

        // If the area 16-31 contains zero's, it concerns waypoints 
        // uploaded by @trip. In this case the name will be equal to the
        // routepoint number (best we can do...). 
        if (data[16]==0x00)
        {
            name=String.format("%03d", routePointNumber);
        }
        else
        {
            // Copy 15 name bytes. Replace \0 with space char
            nameBytes=new byte[15];            i=0;
            while (i<15)
            {
                if (data[16+i]>0)
                {
                    nameBytes[i]=data[16+i];
                }
                else
                {
                    nameBytes[i]=' ';
                }
                i++;
            }
            // create a string from the bytes
            name=new String(nameBytes).trim();
        }
        // Get the symbol ID.
        // For @trip uploaded routes it is 0x00, which result in the default symbol
        // ForuU-gotMe, it result in the real symbol ID
        symbolId=data[31]&0xFF;

        symbol=symbolRepository.getSymbolNameById(symbolId);
    }


    /**
     * Constructor used to initialise the instance with GPX file data
     * @param longitude Longitude
     * @param latitude Latitude
     * @param elevation Elevation
     * @param name Name (max 16 char)
     * @param symbol Symbol name
     */
    public RoutePoint(int longitude, int latitude, int elevation, String name, String symbol)
    {
        super(longitude, latitude, elevation);

        initStatics();

        this.name=name;
        this.symbol=symbol;

        createBitMap();
    }

    /**
     * This method creates the bitmap (as byte array) to be send to the device
     */
    private void createBitMap()
    {
        Graphics2D      g;
        Font            f;
        int             x;
        int             y;
        int             dx;
        int             pixelValue;
        int[]           symbolBitmap;

        int             offset;
        int             style;
        int             fontsize;
        String          fontsizeString;
        Settings        settings;
        

        g=image.createGraphics();

        g.clearRect(0, 0, BITMAP_WIDTH, BITMAP_HEIGHT);

        // Draw the borders and lines
        g.drawLine(0                ,  0, BITMAP_WIDTH-1   , 0              );
        g.drawLine(0                ,  0, 0                , BITMAP_HEIGHT-1);
        g.drawLine(BITMAP_SEPARATORX,  0, BITMAP_SEPARATORX, BITMAP_HEIGHT-1);
        g.drawLine(BITMAP_WIDTH-1   ,  0, BITMAP_WIDTH-1   , BITMAP_HEIGHT-1);

        
        settings=Settings.getInstance();
        fontsizeString=settings.getRoutepointFontsize();
        
        if (fontsizeString.equals("large"))
        {
          offset=1;
          fontsize=18;
          style=Font.BOLD;
        }
        else if (fontsizeString.equals("small"))
        {
          offset=3;
          fontsize=10;
          style=Font.PLAIN;
        }
        else
        {
          offset=2;
          fontsize=14;
          style=Font.PLAIN;
        }
        
        
        // Draw the name of the waypoint
        f = new Font("Lucida Sans", style, fontsize);         // FONT.PLAIN, 10
        g.setFont(f);
        g.drawString(name, BITMAP_NAMEX, BITMAP_HEIGHT-offset);  // BITMAP_HEIGHT-3

        // Draw the symbol
        symbolBitmap=symbolRepository.getSymbolBitmap(symbol);

        if (symbolBitmap!=null)
        {
            y=0;
            while (y<Symbol.SYMBOL_BYTES)
            {
                x=0;
                while (x<15)
                {
                    if ((symbolBitmap[y]&(0x01<<x))!=0)
                    {
                        g.drawLine(x+BITMAP_SYMBOLX, y+1, x+BITMAP_SYMBOLX, y+1);
                    }
                    x++;
                }
                y++;
            }
        }

        // Now convert the image to an array of bytes

        bitmap=new byte[BITMAP_SIZE];
        y=0;
        while (y<BITMAP_HEIGHT)
        {
            x=0;
            while (x<BITMAP_WIDTH/BITS_PER_BYTE)
            {
                dx=0;
                while (dx<BITS_PER_BYTE)
                {
                    if ((image.getRGB(x*BITS_PER_BYTE+dx, y)&0xffffff)==0xFFFFFF)
                    {
                        bitmap[y*(BITMAP_WIDTH/BITS_PER_BYTE)+x]|=(1<<dx);
                    }
                    dx++;
                }
                x++;
            }
            y++;
        }
        
        
    }




    /**
     * This method returns the bitmap. 17 lines of 16 bytes/128 bits
     * @return The bitmap as
     */
    public byte[] getBitmap()
    {
        return bitmap;
    }

    @Override
    public String toString()
    {
        String outputString;

        outputString=new String("Lon: "+longitude+" Lat: "+latitude+
                                " Elevation: "+elevation + " Name: "+name+
                                " Symbol: "+symbol);
        return outputString;
    }




    /**
     * This method returns the RouteWaypoint as a byte array with the
     * format as the Device expects is. The size is 0x120.
     * @return The byte array
     */
    public byte[] getRecordAsByteArray()
    {
        int    lat;
        int    lon;
        int    ele;
        int    i;
        byte[] nameBytes;
        int    nameLength;

        lon=(int)(this.longitude*1e7);
        lat=(int)(this.latitude*1e7);
        ele=(int)(this.elevation*1e7);

        // start of record
        record[ 0]=(byte)0xB8;
        record[ 1]=(byte)0x48;
        record[ 2]=(byte)0x20;
        record[ 3]=(byte)0x02;

        // Latitude
        record[ 4]=(byte)((latitude    )&0xFF);
        record[ 5]=(byte)((latitude>> 8)&0xFF);
        record[ 6]=(byte)((latitude>>16)&0xFF);
        record[ 7]=(byte)((latitude>>24)&0xFF);

        // Longitude
        record[ 8]=(byte)((longitude    )&0xFF);
        record[ 9]=(byte)((longitude>> 8)&0xFF);
        record[10]=(byte)((longitude>>16)&0xFF);
        record[11]=(byte)((longitude>>24)&0xFF);

        // Elevation
        record[12]=(byte)((elevation    )&0xFF);
        record[13]=(byte)((elevation>> 8)&0xFF);
        record[14]=(byte)((elevation>>16)&0xFF);
        record[15]=(byte)((elevation>>24)&0xFF);

        // Piece of empty space (byte 16-31) is used to store the name and symbol

        // Copy the name, fill the rest with 0x00
        nameBytes=name.getBytes();
        nameLength=nameBytes.length;

        // truncate the name to 15 chars
        if (nameLength>15)
        {
            nameLength=15;
        }
        // copy the name
        i=0;
        while (i<nameLength)
        {
            record[16+i]=nameBytes[i];
            i++;
        }
        // Fill the rest with spaces
        while (i<15)
        {
            record[16+i]=(byte)' ';
            i++;
        }

        // Use byte 31 to store the symbol ID
        record[31]=(byte)symbolRepository.getSymbolIdByName(symbol);

        // The bitmap
        i=0;
        while (i<BITMAP_SIZE)
        {
            record[32+i]=bitmap[i];
            i++;
        }
        return record;
    }

    /**
     * This method returns the name of the RoutePoint
     * @return The name
     */
    public String getName()
    {
        return name;
    }

    /**
     * This method returns the name of the symbol
     * @return The name or "" if no name is specified
     */
    public String getSymbolName()
    {
        return symbol;
    }
}
