/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package net.deepocean.u_gotme;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.ImageIcon;
import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.net.URL;

import java.util.ArrayList;
import java.util.Iterator;


/**
 *
 * @author Jorgen
 */
public class MapFrame extends JFrame
{
    // should be 2048 according to google. However, in practice the URL 
    // google processes must be smaller
    private static final int        MAXSTRINGLENGTH=1700;

    private static final int        MAXCOMPRESSIONPARAMS=7;
    private static final int[]      compressionParamPoints     ={  0,     25,    500,   1000,   1500,   2000,   2500,  3000};    
    private static final double[]   compressionParamMaxSlopeDev={0.0,    0.2,    0.3,    0.6,    0.7,    0.8,   0.85,   0.9};    
//    private static final double[]   compressionParamMaxSlopeDev={0.0, 0.0001, 0.00025, 0.0005, 0.0008, 0.0012, 0.0016, 0.002};    
    private int                     compressionParamIndex      =MAXCOMPRESSIONPARAMS-1;

    
    JLabel                      label;
    
   
    Coordinate                  firstPoint;
    Coordinate                  lastPoint;
    int                         totalPointCount;
    int                         compressedPointCount;
    
    
    Coordinate                  previousEncodedPoint=null;

    double                      maxSlopeDeviation=0.0005;
    
    String                      resultString;
    
    public static final String  MAPTYPE_ROAD        ="roadmap";
    public static final String  MAPTYPE_SATELLITE   ="satellite";
    public static final String  MAPTYPE_TERRAIN     ="terrain";
    public static final String  MAPTYPE_HYBRID      ="hybrid";

    private static String       mapType=MAPTYPE_ROAD;
    
    /**
     * Constuctor
     */
    public MapFrame()
    {
        this.setResizable(false);
        JPanel panel = new JPanel();
        
        label = new JLabel();
        

        panel.add(label);
        this.add(panel);
        
//        this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        this.setLocationRelativeTo(null);
        this.setVisible(true);       

    }

    /**
     * Returns the map type currently set
     * @return The map type
     */
    public static String getMapType()
    {
        return mapType;
    }
    
    /**
     * Sets the map type
     * @param newMapType The new map type 
     */
    public static void setMapType(String newMapType)
    {
        if (newMapType.equals(MAPTYPE_ROAD) ||
            newMapType.equals(MAPTYPE_SATELLITE) ||
            newMapType.equals(MAPTYPE_TERRAIN) ||
            newMapType.equals(MAPTYPE_HYBRID))
        {
            mapType=newMapType;
            DebugLogger.info("Map type set to "+mapType);
        }
    }
    
    /**
     * Returns the URL for retreiving the map
     * @return The URL
     */
    private String getMapString()
    {
        String mapString;
        
        mapString="http://maps.googleapis.com/maps/api/staticmap?size=480x480&sensor=false&maptype=" + 
                  mapType;
        

        return mapString;
        
    }
    
    /**
     * This method sets the max slope deviation based on the number of points
     * in a track. It is an empirical relation. The more points we have to process
     * the more we allow the slope to deviate.
     * @param numberOfPoints Number of points in the track.
     */
    void setCompressionParameter(int numberOfPoints)
    {
/*        
        // Setting the 1st to 120 makes sure all route points ()
        if (numberOfPoints<120)
        {
            maxSlopeDeviation=0.0;        
        }
        else if (numberOfPoints<1000)
        {
            maxSlopeDeviation=0.0001;        
        }
        else if (numberOfPoints<2000)
        {
            maxSlopeDeviation=0.0005;        
        }
        else if (numberOfPoints<3000)
        {
            maxSlopeDeviation=0.001;        
        }
        else
        {
            maxSlopeDeviation=0.002;        
        }
*/
        boolean found;
        
        found=false;
        compressionParamIndex=MAXCOMPRESSIONPARAMS-1;
        
        while ((!found) && (compressionParamIndex>0))
        {
            if (numberOfPoints<compressionParamPoints[compressionParamIndex])
            {
                compressionParamIndex--;
            }
            else
            {
                found=true;
            }
        }
        
        maxSlopeDeviation=compressionParamMaxSlopeDev[compressionParamIndex];
        
        DebugLogger.debug("Number of points: "+numberOfPoints+" factor: "+
                          maxSlopeDeviation);
    }
    
    /**
     * This method sets the next higher compression factor, resulting in higher
     * compression.
     * @return True if succeeded, false if no higher compression possible.
     */
    boolean setHigherCompression()
    {
        boolean done;
        
        done=false;
        
        if (compressionParamIndex<MAXCOMPRESSIONPARAMS-1)
        {
            compressionParamIndex++;
            maxSlopeDeviation=compressionParamMaxSlopeDev[compressionParamIndex];

            DebugLogger.debug("Next factor: "+
                              maxSlopeDeviation);            
            done=true;
        }
                
        return done;
    }
    
    
    /**
     * This method resets the compression algorithm.
     */
    void resetCompression()
    {
        firstPoint=null;
        lastPoint=null;
        this.totalPointCount=0;
        this.compressedPointCount=0;
    }
    
    /**
     * This method implements a simple compression algorithm, i.e. it
     * skips intermediate points when the points are more or less on a straight
     * line. Note: call resetCompression() at start, finishCompression() at the
     * end.
     * @param point Point to add
     * @return Point if the point makes up the end result, null if the point
     *         can be skipped
     */
    Coordinate compress(Coordinate point)
    {
        Coordinate      returnPoint;
        Coordinate      midPoint;
        double          x1, y1, xm, ym, x2, y2;
        double          alpha1, alpha2;
        double          averageSlope;
        double          slope;
        double          factor;
        boolean         deviates;
        
        returnPoint=null;
        totalPointCount++;
        
        if (firstPoint==null)
        {
            returnPoint=point;
            firstPoint=point;
        }
        else
        {
            if (lastPoint==null)
            {
                lastPoint=point;
            }
            else
            {
                midPoint=lastPoint;
                lastPoint=point;
                
                x1=firstPoint.getLongitude();
                y1=firstPoint.getLatitude();
                xm=midPoint.getLongitude();
                ym=midPoint.getLatitude();
                x2=lastPoint.getLongitude();
                y2=lastPoint.getLatitude();

                
                factor=Math.sqrt((ym-y1)*(ym-y1)+(xm-x1)*(xm-x1))*maxSlopeDeviation;
                
                deviates=false;

                
                averageSlope=xm-x1;
                slope       =x2-xm;
                
                if ((slope>=averageSlope+factor) || 
                    (slope<=averageSlope-factor))
                {
                    deviates=true;
                }

                averageSlope=ym-y1;
                slope       =y2-ym;
                
                if ((slope>=averageSlope+factor) || 
                    (slope<=averageSlope-factor))
                {
                    deviates=true;
                }

                if (deviates)
                {
                    returnPoint=midPoint;
                    firstPoint=midPoint;

                }
            }
        }

        if (returnPoint!=null)
        {
            compressedPointCount++;
        }
        
        return returnPoint;
    }
    
    /**
     * This method finishes the compression, i.e. it returns the last point
     * that is still in memory.
     * @return 
     */
    Coordinate finishCompression()
    {
        DebugLogger.info("Track compressed: skipped "+(totalPointCount-compressedPointCount)+
                          " points out of "+totalPointCount);
        if (totalPointCount>0)
        {
            DebugLogger.debug("Compressed to "+(100*compressedPointCount/totalPointCount)+
                             "% @ max. allowed slope delta "+this.maxSlopeDeviation);
        }
        return lastPoint;
    }
    

    
    /**
     *  This method resets the Google polyline encoding. Basically it
     *  sets the 1st point to null, indicating that the first point must be
     *  encoded fully (for subsequent points only the delta will be encoded)
     */
    private void resetPointEncoding()
    {
        this.previousEncodedPoint=null;
    }
    
    /**
     * This method encodes a point according the Google encoded polyline
     * method.
     * @param point The point to encode
     * @return String representing the encoded point (or delta with respect
     *                to previous point if it is not the 1st point)
     */
    private String encodePoint(Coordinate point)
    {
        String encodedPointString;
        double deltaLat, deltaLon;
        
        encodedPointString="";
        
        if (previousEncodedPoint==null)
        {
            encodedPointString+=encodeValue(point.getLatitude());
            encodedPointString+=encodeValue(point.getLongitude());
        }
        else
        {
            deltaLat=point.getLatitude()-previousEncodedPoint.getLatitude();
            deltaLon=point.getLongitude()-previousEncodedPoint.getLongitude();
            encodedPointString+=encodeValue(deltaLat);
            encodedPointString+=encodeValue(deltaLon);
        }
        previousEncodedPoint=point;
        
        return encodedPointString;
        
    }
    
    
    /**
     * This method converts doubles to the Google encoded string format
     * @param value Value to convert
     * @return Encoded string part
     */
    private String encodeValue(double value)
    {
        String  conversion;
        int     binValue;
        int     charCode;
        int     nextCharCode;
        char    theChar;
        boolean isNegative;
        int     i;
        boolean finished;
        
        conversion="";
        finished=false;
        
        if (value<0)
        {
            value=-value;
            isNegative=true;
        }
        else
        {
            isNegative=false;
        }
        

        value=value*1e5;
        binValue=(int)Math.round(value);
        if (binValue==0)
        {
            isNegative=false;
        }
        
        if (isNegative)
        {
            binValue=~binValue;
            binValue+=1;
        }
        binValue<<=1;
        
        if (isNegative)
        {
            binValue=~binValue;
        }
        
        i=0;
        while (i<6 && !finished)
        {
            charCode=binValue & 0x1f;
            binValue>>=5;
            
            if (i<5)
            {
                nextCharCode=binValue>>((i+1)*5) & 0x1f;
                if (binValue>0)
                {
                    charCode |= 0x20;
                }
                else
                {
                    finished=true;
                }
            }
            charCode+=63;
            theChar=(char)charCode;
            conversion+=theChar;


            i++;
        }
        
        return conversion;
    }
    
    
    
    String convertTrackMarkers(int trackNo, int maxStringLength)
    {
        String markerString;
        
        markerString="";
        
        return markerString;
    }
    
    
    

    /**
     * This method converts the indicated track to a encoded polyline
     * to be used in the Goolgle map api. All segments are shown in one
     * 'path=...' statement (it is not possible to show multiple paths)
     * The track is compressed in order to reduce the number of points
     * @param trackNo The track to convert
     * @return Full polyline sting.
     */
    String compressAndConvertTrack(int trackNo, int maxStringLength)
    {
        int                         segment;
        int                         numberOfSegments;
        String                      trackString;
        String                      pointString;
        String                      pathString;
        TrackLog                    trackLog;
        ArrayList<TrackLogPoint>    points;
        Iterator<TrackLogPoint>     iterator;
        TrackLogPoint               point;
        Coordinate                  compressedPoint;
        boolean                     bailOut;
        boolean                     found;
        boolean                     maxCompressionReached;
        int                         numberOfTrackPoints;


        trackString     ="";
      
        trackLog        =TrackLog.getInstance();
       
       
        numberOfSegments=trackLog.getNumberOfTrackSegments(trackNo);

        // Calibrate the compression based on the number of trackpoints in
        // the track
        numberOfTrackPoints=0;
        segment=0;
        while (segment<numberOfSegments)
        {
            points=trackLog.getTrackPoints(trackNo, segment);
            numberOfTrackPoints+=points.size();
            segment++;
        }
        
        // Set a default level of compression based on the number of trackpoints.
        this.setCompressionParameter(numberOfTrackPoints);

        // This loop tries higher compression levels, if the compression
        // results in truncated strings.
        maxCompressionReached=false;
        found=false;
        while (!found && !maxCompressionReached)
        {

            bailOut         =false;
            trackString     ="";

            // Process the segments
            segment=0;
            while (segment<numberOfSegments && !bailOut)
            {
                this.resetCompression();
                this.resetPointEncoding();            

                if ((segment%2)>0)
                {
                    pathString="path=color:blue|enc:";
                }
                else
                {
                    pathString="path=color:red|enc:";

                }
                // If adding the path string makes the track string exceed
                // the max length, bail out
                if (trackString.length()+pathString.length()<maxStringLength)
                {
                    trackString+=pathString;
                }
                else
                {
                    bailOut=true;
                }   

                // Retrieve the list of points that make up the segment
                points=trackLog.getTrackPoints(trackNo, segment);

                iterator=points.iterator();

                while (iterator.hasNext() && !bailOut)
                {
                    // Retrieve the next point
                    point           =iterator.next();

                    // Pass it through the compressor
                    compressedPoint=this.compress(point);

                    // If the compressor returns a point, encode it
                    if (compressedPoint!=null)
                    {
                        pointString=this.encodePoint(compressedPoint);

                        // If adding the point string makes the track string exceed
                        // the max length, bail out
                        if (trackString.length()+pointString.length()<maxStringLength)
                        {
                            trackString+=pointString;
                        }
                        else
                        {
                            bailOut=true;
                        }
                    }
                }

                // Process the last point in the segment
                compressedPoint=this.finishCompression();
                if (compressedPoint!=null)
                {
                    pointString=this.encodePoint(compressedPoint);

                    // If adding the point string makes the track string exceed
                    // the max length, bail out
                    if (trackString.length()+pointString.length()<maxStringLength)
                    {
                        trackString+=pointString;
                    }
                    else
                    {
                        bailOut=true;
                    }

                }


                // More segments to follow? Add a pipe symbol as separator

                if (segment<numberOfSegments-1 && !bailOut)
                {
                    trackString+='&';
                }

                segment++;
            }

            if (!bailOut)
            {
                found=true;
            }
            else
            {
                maxCompressionReached=!this.setHigherCompression();
            }
            
        }        
        
        
        
        if (!found)
        {
            resultString="Track truncated for printing. Too much points";
            DebugLogger.info(resultString);            
        }


        return trackString;
    }

    /**
     * This method converts the route to a encoded polyline
     * to be used in the Goolgle map api. 
     * The route is compressed in order to reduce the number of points
     * @return Full polyline sting.
     */
    String compressAndConvertRoute(int maxStringLength)
    {
        String                      routeString;
        String                      pointString;
        String                      pathString;
        String                      markersString;
        RouteLog                    routeLog;
        ArrayList <RoutePoint>      points;
        Iterator<RoutePoint>        iterator;
        RoutePoint                  point;
        Coordinate                  compressedPoint;
        boolean                     bailOut;
        int                         numberOfRoutePoints;


        routeLog            =RouteLog.getInstance();

        numberOfRoutePoints =routeLog.getNumberOfEntries();
        
        this.setCompressionParameter(numberOfRoutePoints);

        bailOut             =false;
        
       
        // Retrieve the list of points that make up the segment
        points=routeLog.getWaypoints();

        
        
        this.resetCompression();
        this.resetPointEncoding();

        // For each segment, create a 'path=...' 
        routeString="path=color:0xff0000|weight:5|enc:";
        
        iterator=points.iterator();

        while (iterator.hasNext() && !bailOut)
        {
            // Retrieve the next point
            point           =iterator.next();

            // Pass it through the compressor
            compressedPoint=this.compress(point);

            // If the compressor returns a point, encode it
            if (compressedPoint!=null)
            {
                pointString=this.encodePoint(compressedPoint);

                // If adding the point string makes the track string exceed
                // the max length, bail out
                if (routeString.length()+pointString.length()<maxStringLength)
                {
                    routeString+=pointString;
                }
                else
                {
                    bailOut=true;
                }
            }
        }            

        // Process the last point in the segment
        compressedPoint=this.finishCompression();
        if (compressedPoint!=null)
        {
            pointString=this.encodePoint(compressedPoint);

            // If adding the point string makes the track string exceed
            // the max length, bail out
            if (routeString.length()+pointString.length()<maxStringLength)
            {
                routeString+=pointString;
            }
            else
            {
                bailOut=true;
            }

        }
            
        if (!bailOut)
        {
            // Show the 1st point 
            markersString="&markers=color:blue|label:S|";
            point=points.get(0);
            markersString+=point.getLatitude()+","+point.getLongitude();
            // Show the last point
            markersString+="&markers=color:green|label:F|";
            point=points.get(points.size()-1);
            markersString+=point.getLatitude()+","+point.getLongitude();
            
            if (routeString.length()+markersString.length()<maxStringLength)
            {
                routeString+=markersString;
            }
        }
            
       
        if (bailOut)
        {
            resultString="Route truncated for printing. Too much points";
            DebugLogger.info(resultString);            
        }


        return routeString;
    }
    
    
    
    
    
    /**
     * This method converts the waypoints in the WaypointLog to a string
     * to be used in Google maps.
     * @param maxStringLength
     * @return 
     */
    private String convertWaypoints(int maxStringLength)
    {
        WaypointLog                 waypointLog;
        ArrayList <Waypoint>        points;
        Iterator<Waypoint>          iterator;
        Waypoint                    point;        
        String                      waypointString;
        String                      markersString;
        boolean                     bailOut;
        
        
        waypointLog=WaypointLog.getInstance();
        
        // Retrieve the list of waypoints
        points=waypointLog.getWaypoints();

        // Parse the points
        iterator=points.iterator();
        
        if (iterator.hasNext())
        {
            markersString="markers=color:blue";
            bailOut=false;


            while (iterator.hasNext() && !bailOut) 
            {
                point           =iterator.next();
                waypointString  ="|"+point.getLatitude()+","+point.getLongitude();
                if (markersString.length()+waypointString.length()<maxStringLength)
                {
                    markersString+=waypointString;
                }
                else
                {
                    resultString="Waypoints truncated, to much waypoints to show on map";
                    DebugLogger.info(resultString);
                    bailOut=true;
                }
            }
        }
        else
        {
            // In fact this should not happen: there must be at least 
            // one waypoint prior to calling this method
            markersString=null;
            resultString="There are no waypoints to show";
            DebugLogger.error(resultString);
        }
        
        return markersString;
    }
    
    
    

    /**
     * This method show the track in this frame on a google map
     * @param trackNo The track to show
     */
    public String showTrack(int trackNo)
    {
        String  trackString;
       
        resultString="Track shown";
        
        trackString=this.getMapString()+"&"; 
        trackString+=compressAndConvertTrack(trackNo, MAXSTRINGLENGTH-trackString.length());
        DebugLogger.debug("Google URL: "+trackString+" Length: "+ trackString.length());        
        BufferedImage image=null;        
        try
        {
            image = ImageIO.read(new URL(trackString));
            label.setIcon(new ImageIcon(image));
            this.pack();
        }
        catch (Exception e)
        {
            resultString="Unable to get Google map"; 
            DebugLogger.error(resultString+": "+e.getMessage()); 
            this.dispose();
        }

        return resultString;
    }
    
    /**
     * This method shows the waypoints on a Google static map
     * @return String describing the result.
     */
    public String showWaypoints()
    {
        String  mapString;
        String  markersString;
       
        resultString="Waypoints shown";
        
        mapString=this.getMapString()+"&"; 
        markersString=convertWaypoints(MAXSTRINGLENGTH-mapString.length());
        mapString+=markersString;
        if (markersString!=null)
        {
        DebugLogger.debug("Google URL: "+mapString+" Length: "+ mapString.length());   

            BufferedImage image=null;        
            try
            {
                image = ImageIO.read(new URL(mapString));
                label.setIcon(new ImageIcon(image));
                this.pack();
            }
            catch (Exception e)
            {
                resultString="Unable to get Google map"; 
                DebugLogger.error(resultString+": "+e.getMessage()); 
                this.dispose();
            }
        }
        
        return resultString;
    }
    
    /**
     * This method shows the route (uploaded waypoints)
     * @return String describing the results
     */
    String showRoute()
    {
        String          trackString;
        
        resultString="Route shown";
        
        trackString=this.getMapString()+"&"; 
        trackString+=compressAndConvertRoute(MAXSTRINGLENGTH-trackString.length());
        DebugLogger.debug("Google URL: "+trackString+" Length: "+ trackString.length());         
        BufferedImage image=null;        
        try
        {
            image = ImageIO.read(new URL(trackString));
            label.setIcon(new ImageIcon(image));
            this.pack();
        }
        catch (Exception e)
        {
            resultString="Unable to get Google map"; 
            DebugLogger.error(resultString+": "+e.getMessage()); 
            this.dispose();
        }
        
        
        return resultString;
    }
    
}
