/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import java.io.File;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Comment;
import org.w3c.dom.Element;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;

import hirondelle.date4j.DateTime;

/**
 * This class writes tracks and waypoints to GPX file.
 * The major difference between GPX 1.0 and GPX 1.1 is the <extensions> element.
 * This element is available in GPX 1.1 and not in GPX 1.0.
 * Minor difference is that <speed> and <course> no longer exist in GPX 1.1.
 * Therefore, tracks writen in GPX 1.1 use the extensions element. Private
 * fields are used in the <extensions> element in the u-gotMe namespace
 * @author Jorgen
 */
public class GpxWriter implements IgotuWriter
{
    private static GpxWriter    theInstance=null;

    private TrackLog            trackLog;
    private WaypointLog         waypointLog;
    private RouteLog            routeLog;
    private int                 trackPoints;
    private int                 wayPoints;
    private String              gpxVersion;


    Document                    doc;
    Element                     gpxElement;


    /**
     * Constructor
     */
    private GpxWriter()
    {
        trackLog        =TrackLog.getInstance();
        waypointLog     =WaypointLog.getInstance();
        routeLog        =RouteLog.getInstance();
        gpxVersion      =new String("1.1");
    }

    /**
     * This method returns the one and only instance of this singleton class
     * @return The instance
     */
    public static GpxWriter getInstance()
    {
        if (theInstance==null)
        {
            theInstance=new GpxWriter();
        }

        return theInstance;
    }

    /**
     * This method sets the GPX version
     * @param newVersion The new version "1.0" and "1.1" allowed.
     */
    public void setGpxVersion(String newVersion)
    {
        if(newVersion.equals("1.0") || newVersion.equals("1.1"))
        {
            this.gpxVersion=newVersion;
        }
        else
        {
            DebugLogger.error("Illegal GPX version "+newVersion+
                              ". Version left to "+gpxVersion);
        }
    }


    /**
     * This method creates the XML document, adds the GPX headers and
     * creates the <gpx> element. The variables doc and gpxElement will
     * be global variables in this class.
     * @throws javax.xml.parsers.ParserConfigurationException
     */
    private void createGpxDocument(Device.DeviceType deviceType) throws ParserConfigurationException
    {
        String      creator;


        DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder docBuilder = docFactory.newDocumentBuilder();

        // The document
        doc = docBuilder.newDocument();


        // gpx elements
        gpxElement = doc.createElement("gpx");
        doc.appendChild(gpxElement);


        creator=new String("u-gotMe - ")+
        Device.getDeviceTypeDescription(deviceType);
        if (gpxVersion.equals("1.0"))
        {
            this.addGpx1_0Header(doc, gpxElement, creator);
        }
        else if (gpxVersion.equals("1.1"))
        {
            this.addGpx1_1Header(doc, gpxElement, creator);
        }
    }

    /**
     * Add the GPX 1.0 header
     * @param doc The XML document
     * @param gpxElement The GPX element
     */
    private void addGpx1_0Header(Document doc, Element gpxElement, String creator)
    {
        Attr        attr;

        // GPX version 1.0

        attr = doc.createAttribute("creator");
        attr.setValue(creator);
        gpxElement.setAttributeNode(attr);

        attr = doc.createAttribute("version");
        attr.setValue("1.0");
        gpxElement.setAttributeNode(attr);

        // GPX namespace
        attr = doc.createAttribute("xmlns");
        attr.setValue("http://www.topografix.com/GPX/1/0");
        gpxElement.setAttributeNode(attr);

        // XMLSchema namespace
        attr = doc.createAttribute("xmlns:xsi");
        attr.setValue("http://www.w3.org/2001/XMLSchema-instance");
        gpxElement.setAttributeNode(attr);

         // Schema locations - just the GPX location
        attr = doc.createAttribute("xsi:schemaLocation");
        attr.setValue("http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd ");
        gpxElement.setAttributeNode(attr);

    }

    /**
     * Add the GPX 1.0 header
     * @param doc The XML document
     * @param gpxElement The GPX element
     */
    private void addGpx1_1Header(Document doc, Element gpxElement, String creator)
    {
        Comment     comment;
        Attr        attr;

        // GPX version 1.1
        attr = doc.createAttribute("creator");
        attr.setValue(creator);
        gpxElement.setAttributeNode(attr);

        attr = doc.createAttribute("version");
        attr.setValue("1.1");
        gpxElement.setAttributeNode(attr);

        // XMLSchema namespace
        attr = doc.createAttribute("xmlns:xsi");
        attr.setValue("http://www.w3.org/2001/XMLSchema-instance");
        gpxElement.setAttributeNode(attr);

        // GPX namespace
        attr = doc.createAttribute("xmlns");
        attr.setValue("http://www.topografix.com/GPX/1/1");
        gpxElement.setAttributeNode(attr);

        // u-gotMe namespace
        attr = doc.createAttribute("xmlns:u-gotMe");
        attr.setValue("http://u-gotme.deepocean.net");
        gpxElement.setAttributeNode(attr);

        // Schema locations
        attr = doc.createAttribute("xsi:schemaLocation");
        attr.setValue("http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd "+
                      "http://u-gotme.deepocean.net http://www.deepocean.net/u-gotme/u-gotme.xsd");
        gpxElement.setAttributeNode(attr);

    }

    /**
     * This method writes the GPX file
     * @param fileName Name of the file
     * @throws javax.xml.transform.TransformerException
     */
    void writeGpxDocument(String fileName) throws TransformerException
    {
            // write the content into xml file
            TransformerFactory transformerFactory = TransformerFactory.newInstance();
            transformerFactory.setAttribute("indent-number", 4);

            Transformer transformer = transformerFactory.newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");


            DOMSource source = new DOMSource(doc);
            StreamResult result = new StreamResult(new File(fileName));


            transformer.transform(source, result);
    }

    /**
     * Appends the track points to the segment under GPX 1.0. GPX 1.0 supports
     * the elements 'course' and 'speed'. Under 'hdop' the ehpe value is
     * stored. In fact this is not correct, since ehpe is not hdop.
     * @param doc
     * @param segmentElement Document element representing the track segment
     * @param trackNo The track
     * @param segmentNo The segment
     */
    private void appendTrackGpx1_0(Document doc, Element segmentElement, int trackNo, int segmentNo)
    {
        ArrayList <TrackLogPoint>   points;
        Iterator<TrackLogPoint>     iterator;
        TrackLogPoint               point;
        Element                     pointElement;
        Element                     element;
        Attr                        attr;
        DateTime                    dateTime;
        String                      dateTimeString;


        points=trackLog.getTrackPoints(trackNo, segmentNo);

        iterator=points.iterator();

        while (iterator.hasNext())
        {
            point           =iterator.next();
            pointElement    = doc.createElement("trkpt");
            segmentElement.appendChild(pointElement);

            element    = doc.createElement("ele");
            element.appendChild(doc.createTextNode(String.valueOf(point.getElevation())));
            pointElement.appendChild(element);

            element    = doc.createElement("time");
            dateTime=point.getDateTime();
            dateTimeString=dateTime.format("YYYY-MM-DD")+"T"+
                           dateTime.format("hh:mm:ss")+"Z";
            element.appendChild(doc.createTextNode(dateTimeString));
            pointElement.appendChild(element);
           
            // Extensions: course
            element    = doc.createElement("course");
            element.appendChild(doc.createTextNode(String.valueOf(point.getCourse())));
            pointElement.appendChild(element);            

            // Extensions: speed
            element    = doc.createElement("speed");
            element.appendChild(doc.createTextNode(String.valueOf(point.getSpeed())));
            pointElement.appendChild(element);

            element    = doc.createElement("sat");
            element.appendChild(doc.createTextNode(String.valueOf(point.getSatelliteNumber())));
            pointElement.appendChild(element);
/*
            element    = doc.createElement("hdop");
            element.appendChild(doc.createTextNode(String.valueOf(point.getEhpe())));
            pointElement.appendChild(element);
*/

            // set attribute 'lat' to element
            attr = doc.createAttribute("lat");
            attr.setValue(String.valueOf(point.getLatitude()));
            pointElement.setAttributeNode(attr);

            // set attribute 'lon' to element
            attr = doc.createAttribute("lon");
            attr.setValue(String.valueOf(point.getLongitude()));
            pointElement.setAttributeNode(attr);

            trackPoints++;

        }
    }

    /**
     * Appends the track points to the segment under GPX 1.1. GPX 1.1
     * does not support the elements 'course' and 'speed'.
     * The elements 'course', 'speed' and 'ehpe' are stored under the
     * 'extensions' element.
     * @param doc
     * @param segmentElement Document element representing the track segment
     * @param trackNo The track
     * @param segmentNo The segment
     */
    private void appendTrackGpx1_1(Document doc, Element segmentElement, int trackNo, int segmentNo)
    {
        ArrayList <TrackLogPoint>   points;
        Iterator<TrackLogPoint>     iterator;
        TrackLogPoint               point;
        Element                     pointElement;
        Element                     element;
        Element                     extensionsElement;
        Attr                        attr;
        DateTime                    dateTime;
        String                      dateTimeString;


        points=trackLog.getTrackPoints(trackNo, segmentNo);

        iterator=points.iterator();

        while (iterator.hasNext())
        {
            point           =iterator.next();
            pointElement    = doc.createElement("trkpt");
            segmentElement.appendChild(pointElement);

            // The elevation. For Pro versions it is the height sensor value. 
            // For non-Pro versions it is measured by GPS.
            element    = doc.createElement("ele");
            element.appendChild(doc.createTextNode(String.valueOf(point.getElevation())));
            pointElement.appendChild(element);

            element    = doc.createElement("time");
            dateTime=point.getDateTime();
            dateTimeString=dateTime.format("YYYY-MM-DD")+"T"+
                           dateTime.format("hh:mm:ss")+"Z";
            element.appendChild(doc.createTextNode(dateTimeString));
            pointElement.appendChild(element);

            element    = doc.createElement("sat");
            element.appendChild(doc.createTextNode(String.valueOf(point.getSatelliteNumber())));
            pointElement.appendChild(element);

            extensionsElement    = doc.createElement("extensions");
            pointElement.appendChild(extensionsElement);

            // Extensions: course
            element    = doc.createElement("u-gotMe:course");
            element.appendChild(doc.createTextNode(String.valueOf(point.getCourse())));
            extensionsElement.appendChild(element);

            // Extensions: speed
            element    = doc.createElement("u-gotMe:speed");
            element.appendChild(doc.createTextNode(String.valueOf(point.getSpeed())));
            extensionsElement.appendChild(element);

            // Extensions: ehpe
            element    = doc.createElement("u-gotMe:ehpe");
            element.appendChild(doc.createTextNode(String.valueOf(point.getEhpe())));
            extensionsElement.appendChild(element);

            // For the Pro also log the gps measured elevation
            if ((point.getDeviceType()==Device.DeviceType.DEVICETYPE_GT800PRO) ||
                (point.getDeviceType()==Device.DeviceType.DEVICETYPE_GT820PRO) ||
                (point.getDeviceType()==Device.DeviceType.DEVICETYPE_GT900PRO))    
            {
                // Extensions: ehpe
                element    = doc.createElement("u-gotMe:elegps");
                element.appendChild(doc.createTextNode(String.valueOf(point.getElevationByGps())));
                extensionsElement.appendChild(element);
            }

            // Extensions: ehpe
            element    = doc.createElement("u-gotMe:heartrate");
            element.appendChild(doc.createTextNode(String.valueOf(point.getHeartRate())));
            extensionsElement.appendChild(element);

            
            
            // set attribute 'lat' to element
            attr = doc.createAttribute("lat");
            attr.setValue(String.valueOf(point.getLatitude()));
            pointElement.setAttributeNode(attr);

            // set attribute 'lon' to element
            attr = doc.createAttribute("lon");
            attr.setValue(String.valueOf(point.getLongitude()));
            pointElement.setAttributeNode(attr);

            trackPoints++;
        }
    }



    /**
     * Appends the waypoints belonging to the given track segment.
     * Assume GPX 1.0
     * @param doc The XML document
     * @param trackElement The element representing the track
     * @param trackNo The track identification
     * @param segmentNo The segment identification
     */
    private void appendWaypointsGpx1_0(Document doc, Element trackElement, int trackNo, int segmentNo)
    {
        ArrayList <TrackLogPoint>   points;
        Iterator<TrackLogPoint>     iterator;
        TrackLogPoint               point;
        Element                     pointElement;
        Element                     element;
        Element                     extensionElement;
        Attr                        attr;
        DateTime                    dateTime;
        String                      dateTimeString;


        // Retrieve the list of waypoints
        points=trackLog.getWayPoints(trackNo, segmentNo);

        // Parse the points
        iterator=points.iterator();

        while (iterator.hasNext())
        {
            point           =iterator.next();
            pointElement    = doc.createElement("wpt");
            trackElement.appendChild(pointElement);

            element    = doc.createElement("ele");
            element.appendChild(doc.createTextNode(String.valueOf(point.getElevation())));
            pointElement.appendChild(element);

            element    = doc.createElement("time");
            dateTime=point.getDateTime();
            dateTimeString=dateTime.format("YYYY-MM-DD")+"T"+
                           dateTime.format("hh:mm:ss")+"Z";
            element.appendChild(doc.createTextNode(dateTimeString));
            pointElement.appendChild(element);
            
            element    = doc.createElement("name");
            element.appendChild(doc.createTextNode(String.valueOf(wayPoints)));
            pointElement.appendChild(element);
           
            element    = doc.createElement("sym");
            element.appendChild(doc.createTextNode("Waypoint"));
            pointElement.appendChild(element);
           
            element    = doc.createElement("sat");
            element.appendChild(doc.createTextNode(String.valueOf(point.getSatelliteNumber())));
            pointElement.appendChild(element);
/*
            element    = doc.createElement("hdop");
            element.appendChild(doc.createTextNode(String.valueOf(point.getEhpe())));
            pointElement.appendChild(element);
*/
            // set attribute 'lat' to element
            attr = doc.createAttribute("lat");
            attr.setValue(String.valueOf(point.getLatitude()));
            pointElement.setAttributeNode(attr);

            // set attribute 'lon' to element
            attr = doc.createAttribute("lon");
            attr.setValue(String.valueOf(point.getLongitude()));
            pointElement.setAttributeNode(attr);
            
            wayPoints++;
        }

    }

    /**
     * Appends the waypoints belonging to the given track segment.
     * Assume GPX 1.1
     * @param doc The XML document
     * @param trackElement The element representing the track
     * @param trackNo The track identification
     * @param segmentNo The segment identification
     */
    private void appendWaypointsGpx1_1(Document doc, Element trackElement, int trackNo, int segmentNo)
    {
        ArrayList <TrackLogPoint>   points;
        Iterator<TrackLogPoint>     iterator;
        TrackLogPoint               point;
        Element                     pointElement;
        Element                     element;
        Element                     extensionsElement;
        Attr                        attr;
        DateTime                    dateTime;
        String                      dateTimeString;


        // Retrieve the list of waypoints
        points=trackLog.getWayPoints(trackNo, segmentNo);

        // Parse the points
        iterator=points.iterator();

        while (iterator.hasNext())
        {
            point           =iterator.next();
            pointElement    = doc.createElement("wpt");
            trackElement.appendChild(pointElement);

            element    = doc.createElement("ele");
            element.appendChild(doc.createTextNode(String.valueOf(point.getElevation())));
            pointElement.appendChild(element);

            element    = doc.createElement("time");
            dateTime=point.getDateTime();
            dateTimeString=dateTime.format("YYYY-MM-DD")+"T"+
                           dateTime.format("hh:mm:ss")+"Z";
            element.appendChild(doc.createTextNode(dateTimeString));
            pointElement.appendChild(element);

            element    = doc.createElement("name");
            element.appendChild(doc.createTextNode(String.valueOf(wayPoints)));
            pointElement.appendChild(element);

            element    = doc.createElement("sym");
            element.appendChild(doc.createTextNode("Waypoint"));
            pointElement.appendChild(element);

            element    = doc.createElement("sat");
            element.appendChild(doc.createTextNode(String.valueOf(point.getSatelliteNumber())));
            pointElement.appendChild(element);

            extensionsElement    = doc.createElement("extensions");
            pointElement.appendChild(extensionsElement);

            element    = doc.createElement("u-gotMe:ehpe");
            element.appendChild(doc.createTextNode(String.valueOf(point.getEhpe())));
            extensionsElement.appendChild(element);


            // set attribute 'lat' to element
            attr = doc.createAttribute("lat");
            attr.setValue(String.valueOf(point.getLatitude()));
            pointElement.setAttributeNode(attr);

            // set attribute 'lon' to element
            attr = doc.createAttribute("lon");
            attr.setValue(String.valueOf(point.getLongitude()));
            pointElement.setAttributeNode(attr);

            wayPoints++;
        }

    }


    /**
     * This method adds the track segments to the track.
     * @param doc XML document
     * @param gpxElement The GPX element
     * @param trackNo The track identification
     * @param trackName The track name
     */
    private void addTrack(Document doc, Element gpxElement, int trackNo, String trackName)
    {
        int     i;
        int     numberOfSegments;
        Track   track;
        Element trackElement;
        Element element;
        String  description;

        numberOfSegments=trackLog.getNumberOfTrackSegments(trackNo);

        // Add the waypoints of all segment. They are appended right under
        // the <gpx> element
        i=0;
        while (i<numberOfSegments)
        {
            // segment
            
            if (gpxVersion.equals("1.0"))
            {
                appendWaypointsGpx1_0(doc, gpxElement, trackNo, i);
            }
            else if (gpxVersion.equals("1.1"))
            {
                appendWaypointsGpx1_1(doc, gpxElement, trackNo, i);
            }

            i++;
        }

        // The track element
        trackElement = doc.createElement("trk");
        gpxElement.appendChild(trackElement);

        element = doc.createElement("name");
        element.appendChild(doc.createTextNode(trackName));
        trackElement.appendChild(element);

        description="i-gotu "+ Device.getDeviceTypeDescription(trackLog.getDeviceType()) +" logged track";
        element = doc.createElement("desc");
        element.appendChild(doc.createTextNode(description));
        trackElement.appendChild(element);        
        
        // Add the track segments.
        i=0;
        while (i<numberOfSegments)
        {
            // segment
            Element segmentElement = doc.createElement("trkseg");
            trackElement.appendChild(segmentElement);

            if (gpxVersion.equals("1.0"))
            {
                appendTrackGpx1_0(doc, segmentElement, trackNo, i);
            }
            else if (gpxVersion.equals("1.1"))
            {
                appendTrackGpx1_1(doc, segmentElement, trackNo, i);
            }

            
            i++;
        }
    }







    /**
     * This method adds the waypoints from the WaypointLog to the GPX
     * structure. Should be compatible with GPX 1.0 and GPX 1.1
     * @param doc The document
     * @param gpxElement The GPX element
     */
    private void addWaypointsFromLog(Document doc, Element gpxElement)
    {
        
        ArrayList <Waypoint>        points;
        Enumeration<Waypoint>       enumerator;
        Iterator<Waypoint>          iterator;
        Waypoint                    point;
        Element                     pointElement;
        Element                     element;
        Attr                        attr;
        DateTime                    dateTime;
        String                      dateTimeString;


        points=waypointLog.getWaypoints();
        
        // Parse the points
        iterator=points.iterator();

        while (iterator.hasNext())
        {
            point           =iterator.next();
            pointElement    = doc.createElement("wpt");
            gpxElement.appendChild(pointElement);

            element    = doc.createElement("ele");
            element.appendChild(doc.createTextNode(String.valueOf(point.getElevation())));
            pointElement.appendChild(element);

            element    = doc.createElement("time");
            dateTime=point.getDateTime();
            dateTimeString=dateTime.format("YYYY-MM-DD")+"T"+
                           dateTime.format("hh:mm:ss")+"Z";
            element.appendChild(doc.createTextNode(dateTimeString));
            pointElement.appendChild(element);

            element    = doc.createElement("name");
            element.appendChild(doc.createTextNode(String.valueOf(point.getId())));
            pointElement.appendChild(element);

            element    = doc.createElement("sym");
            element.appendChild(doc.createTextNode("Waypoint"));
            pointElement.appendChild(element);


            // set attribute 'lat' to element
            attr = doc.createAttribute("lat");
            attr.setValue(String.valueOf(point.getLatitude()));
            pointElement.setAttributeNode(attr);

            // set attribute 'lon' to element
            attr = doc.createAttribute("lon");
            attr.setValue(String.valueOf(point.getLongitude()));
            pointElement.setAttributeNode(attr);

            wayPoints++;
        }

    }




    /**
     * This method adds the waypoints from the WaypointLog to the GPX
     * structure. Should be compatible with GPX 1.0 and GPX 1.1
     * @param doc The document
     * @param gpxElement The GPX element
     */
    private void addRoutePointsFromLog(Element routeElement)
    {

        ArrayList <RoutePoint>      points;
        Iterator<RoutePoint>        iterator;
        RoutePoint                  point;
        Element                     pointElement;
        Element                     element;
        Attr                        attr;



        points=routeLog.getWaypoints();

        // Parse the points
        iterator=points.iterator();

        while (iterator.hasNext())
        {
            point           =iterator.next();

            
            pointElement    = doc.createElement("rtept");
            routeElement.appendChild(pointElement);

            // set attribute 'lat' to element
            attr = doc.createAttribute("lat");
            attr.setValue(String.valueOf(point.getLatitude()));
            pointElement.setAttributeNode(attr);

            // set attribute 'lon' to element
            attr = doc.createAttribute("lon");
            attr.setValue(String.valueOf(point.getLongitude()));
            pointElement.setAttributeNode(attr);

            element    = doc.createElement("ele");
            element.appendChild(doc.createTextNode(String.valueOf(point.getElevation())));
            pointElement.appendChild(element);

            element    = doc.createElement("name");
            element.appendChild(doc.createTextNode(String.valueOf(point.getName())));
            pointElement.appendChild(element);


            element    = doc.createElement("sym");
            element.appendChild(doc.createTextNode(point.getSymbolName()));
            pointElement.appendChild(element);

            wayPoints++;
        }

    }




    /* ************************************************************************\
     * The interface functions
     * ************************************************************************/
    /**
     * Write the track to a GPX file
     * @param fileName Name of the file to write to
     */
    public void writeTrackToFile(String fileName, int trackNo, String trackName)
    {
        Element     trackElement;
        Element     element;
        Comment     comment;
        Attr        attr;
        String      creator;


        wayPoints=0;
        trackPoints=0;

        try
        {
            // create the GPX file
            createGpxDocument(trackLog.getDeviceType());


            addTrack(doc, gpxElement, trackNo, trackName);

            // write the content into xml file
            writeGpxDocument(fileName);

            DebugLogger.info("GpxWriter says: 'File saved to " + fileName + "!'");
            DebugLogger.info("Track: "+trackName+", track points: "+trackPoints+
                             ", wayPoints: "+wayPoints);

        }
        catch (ParserConfigurationException pce)
        {
            pce.printStackTrace();
        }
        catch (TransformerException tfe)
        {
            tfe.printStackTrace();
        }

    }


   /**
     * Write the track to a GPX file
     * @param fileName Name of the file to write to
     */
    public void writeWaypointsToFile(String fileName)
    {
        Element     trackElement;
        Element     element;
        Comment     comment;
        Attr        attr;
        String      creator;


        wayPoints=0;

        try
        {

            this.createGpxDocument(waypointLog.getDeviceType());

            addWaypointsFromLog(doc, gpxElement);


            // write the content into xml file
            writeGpxDocument(fileName);

            DebugLogger.info("GpxWriter says: 'File saved to " + fileName + "!'");
            DebugLogger.info("Added waypoints: "+wayPoints);

        }
        catch (ParserConfigurationException pce)
        {
            pce.printStackTrace();
        }
        catch (TransformerException tfe)
        {
            tfe.printStackTrace();
        }

    }


    /**
     * The method writes the route in the RouteLog to file
     * @param fileName The name of the file to write to
     */
    public void writeRouteToFile(String fileName)
    {

        Element     element;
        Element     routeElement;
        Comment     comment;
        Attr        attr;


        wayPoints=0;

        try
        {
            // create the GPX file
            createGpxDocument(routeLog.getDeviceType());

            // Create the route element
            routeElement= doc.createElement("rte");
            gpxElement.appendChild(routeElement);

            element    = doc.createElement("name");
            element.appendChild(doc.createTextNode("route"));
            routeElement.appendChild(element);

            element    = doc.createElement("src");
            element.appendChild(doc.createTextNode("u-gotMe"));
            routeElement.appendChild(element);


            // Add the route points
            addRoutePointsFromLog(routeElement);

            // write the content into xml file
            writeGpxDocument(fileName);

            DebugLogger.info("GpxWriter says: 'File saved to " + fileName + "!'");
            DebugLogger.info("Added route points: "+wayPoints);

        }
        catch (ParserConfigurationException pce)
        {
            pce.printStackTrace();
        }
        catch (TransformerException tfe)
        {
            tfe.printStackTrace();
        }
    }

    /**
     * The method writes the heartrate  of one track to file
     * @param fileName The name of the file to write to
     */
    public void writeHeartRateToFile(String fileName, int trackNo)
    {
        
    }
}

