/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;


import java.io.*;
import java.util.ArrayList;
import java.util.Iterator;



/**
 * This class represents the i-gotu device. It offers functionality
 * to access the data and information in the device
 * @author Jorgen
 */
public class Device
{
    public enum     ModelType
                    {
                        MODELTYPE_GT100,
                        MODELTYPE_GT120,
                        MODELTYPE_GT200,
                        MODELTYPE_GT800PLUS,
                        MODELTYPE_UNKNOWN
                    }
    
    public enum    DeviceType
            {
                DEVICETYPE_GT100,
                DEVICETYPE_GT120,
                DEVICETYPE_GT200,
                DEVICETYPE_GT800,
                DEVICETYPE_GT800PRO,
                DEVICETYPE_GT820,
                DEVICETYPE_GT820PRO,
                DEVICETYPE_GT900,
                DEVICETYPE_GT900PRO,
                DEVICETYPE_GT800_OR_GT800PRO,
                DEVICETYPE_UNKNOWN
            }

    /** The one and only instane of this class (singleton patern) */
    static Device                       theInstance=null;
    
    /** Connection to the device */
    private final Connection            connection;
    
    /** Buffer containing the data to be written to the device */
    private final byte[]                writeData;

    
    
    /** The response header, you know, the 0x93 0xXX 0xYY thing */
    private final byte[]                responseHeader;

    /** The response data */
    private final byte[]                responseData;


    private int                         trackRecordCount;

    private int                         logInterval;

    /** The modeltype */
    private ModelType                   modelType;
    
    /** The exact device type */
    private DeviceType                  deviceType;

    /** Serial number of the device */
    private int                         serialNumber;
    
    /** The usb version */
    private int                         usbVersion;
    
    private int                         softwareVersionMajor;
    private int                         softwareVersionMinor;



    /** Result of a device access */
    private boolean                     isError;

    /** Memory dependant variables */
    private MemoryMap                   memMap;

    /** Indicates whether caching is enabled for the track log */
    private boolean                     isCaching;       
    
    /** The cache, in case of cache enabled */
    private TrackMemoryCache            cache;
    
    
    /**
     * Constructor. Allocates the data, initialises the required variables.
     * @author Jorgen van der Velde
     */
    private Device()
    {
        Settings settings;
        String   path;
        
        settings=Settings.getInstance();
        
        if (settings.isSimulationMode())
        {
            path                =settings.getSimulationPath();
            connection          =(Connection)new ConnectionSimulation(path);
        }
        else
        {
            if (settings.getComportLib().equals("jssc"))
            {
                DebugLogger.info("Using JSSC comport lib");
                connection      =(Connection)new ConnectionSerialJssc();                
            }
            else if (settings.getComportLib().equals("purejava"))
            {
                DebugLogger.info("Using Purejava comport lib");
                connection      =(Connection)new ConnectionSerialPurejava();                
            }
            else
            {
                DebugLogger.info("Using RXTX comport lib");
                connection      =(Connection)new ConnectionSerialRxtx();                
            }
        }
        responseHeader          =connection.getResponseHeader();
        responseData            =connection.getResponseData();
        

        writeData               =new byte[Connection.BLOCK_SIZE];

        modelType               =ModelType.MODELTYPE_UNKNOWN;
        deviceType              =DeviceType.DEVICETYPE_UNKNOWN;
        serialNumber            =0;
        usbVersion              =0;
        
        logInterval             =-1;

        // At this moment it is initialised here. It should be intialised 
        // based on the device type. Set it to the base class, which is false
        memMap                  =new MemoryMap();
        
        isCaching               =settings.getTrackCaching();
        cache                   =null;
    }

    
    public Connection getConnection()
    {
        return this.connection;
    }
    
    /**
     * Singleton function to get the one and only instance of the device
     * @return The one and only instance of this class.
     * @author Jorgen van der Velde
     */
    public static Device getInstance()
    {
        if (theInstance==null)
        {
            theInstance=new Device();
        }
        
        return theInstance;
    }

   /**
     * Opens connection to the device
     * @param comport Number of the comport to which the igotu is attached
     * @author Jorgen van der Velde
     */
 
    public void open(String comport)
    {
        connection.open(comport);
    }
    
    /**
     * This method closes the comport 
     */
    public void close()
    {
        connection.close();
    }
    
    
    /**
     * Retrieves device info: serial no, version and model
     * @return String containing the result
     * @author Jorgen van der Velde
     */
    String commandGetIdentification()
    {
        String outputString;
        int     dataLength;
        
        DebugLogger.info("* Getting device identfication");
        
        dataLength=connection.commandGetIdentification();
        
        if (dataLength==10)
        {
            serialNumber=((responseData[0]&0xff))+
                         ((responseData[1]&0xff)<<8)+
                         ((responseData[2]&0xff)<<16)+
                         ((responseData[3]&0xff)<<24);
            usbVersion=((responseData[8]&0xff))+
                         ((responseData[9]&0xff)<<8);
            this.softwareVersionMajor=responseData[4]&0xff;
            this.softwareVersionMinor=responseData[5]&0xff;
            outputString="Serial: " + serialNumber +
                         " Version: " + this.softwareVersionMajor + "." +
                                        String.format("%02d", this.softwareVersionMinor) +
                         " Model: " + (((responseData[6]&0xff)<<8) + (responseData[7]&0xff))+
                         " USB lib version: "+usbVersion+"\n";
        }
        else
        {
            outputString="Response not as expected\n";
        }
        return outputString;    
    }

    /**
     * Retrieves model info
     * @author Jorgen van der Velde
     */
    String commandGetModel()
    {
        String  outputString;
        int     deviceId;
        int     unknown;
        int     model;
        int     dataLength;


        isError=false;

        DebugLogger.info("* Getting device model info");
        
        dataLength=connection.commandGetModel();
        
        if (dataLength==3)
        {
            deviceId=responseData[2];
            model=deviceId;

            // always 0xc220
            unknown=(responseData[0]&0xff)+((responseData[1]&0xff)<<8);
            outputString="Device type: "+deviceId+" - ";
            switch(deviceId)
            {
// Non supported devices                
/*
                case 0x13:
                    modelType=ModelType.MODELTYPE_GT100;
                    outputString+="GT-100";
                    break;
                case 0x14:
                    modelType=ModelType.MODELTYPE_GT120;
                    outputString+="GT-120";
                    break;
                case 0x15:
                    modelType=ModelType.MODELTYPE_GT200;
                    outputString+="GT-200";
                    break;
                case 0x16:
                    modelType=ModelType.MODELTYPE_UNKNOWN;
                    outputString+="unknown";
                    break;
*/                
                // Supported devices
                case 0x15:
                    modelType=ModelType.MODELTYPE_GT120;
                    outputString+="GT-120";
                    break;
                case 0x17:
                    modelType=ModelType.MODELTYPE_GT800PLUS;
                    outputString+="GT-800, GT-820 or GT-900 (Pro)";
                    break;
                default:
                    modelType=ModelType.MODELTYPE_UNKNOWN;
                    outputString+="unknown";
                    break;
            }
            outputString+="\n";
        }
        
        else
        {
            isError=true;
            outputString="Response not as expected\n";
        }

        return outputString;
    }

    /**
     * Sets the NMEA mode.
     * @param mode 00 - dongle mode 01 - tracer mode 03 - configure mode
     * @author Jorgen van der Velde
     */
    String commandNmeaSwitch(byte mode)
    {
        String  outputString;
        int     deviceId;
        int     dataLength;

        isError=false;

        DebugLogger.info("* Setting NMEA switch mode to "+mode);
        
        if (mode==0 || mode==1 || mode==3)
        {
            dataLength=connection.commandNmeaSwitch(mode);

            if (dataLength==0)
            {
                outputString="Switched to mode " + mode;
            }
            else
            {
                outputString="Response not as expected\n";
                isError=true;
            }
        }
        else
        {
            outputString="Invalid nmea mode: " + mode;
        }
        return outputString;
    }

    /**
     * Retrieves number of track records, valid and invalid
     * As a side effect the local variable trackRecordCount is set.
     * @return String indicating the result of the function
     * @author Jorgen van der Velde
     */
    public String commandGetCount()
    {
        String  outputString;
        int     count;
        int     dataLength;
        int     maxTrackPoints;

        isError=false;
        maxTrackPoints=memMap.getTracksMaxRecords();

        DebugLogger.info("* Getting number of logged datapoints from device");

        dataLength=connection.commmandCount();

        if (dataLength==3)
        {
            count=((responseData[0]&0xff)<<16)+((responseData[1]&0xff)<<8)+(responseData[2]&0xff);
            outputString="Number of datapoints: " + count + " out of " + maxTrackPoints + 
                         " ("+ 100*(maxTrackPoints-count)/maxTrackPoints+ "% free";
                    
            if (logInterval>0)
            {
                outputString+=", "+ ((maxTrackPoints-count)*logInterval/3600)+
                              " hours of logging left @ ";
                
                if (logInterval>=60)
                {
                    outputString+=logInterval/60 +" min log interval";
                }
                else
                {
                    outputString+=logInterval +" sec log interval";
                }
            }
            outputString+=")\n";
            trackRecordCount=count;
        }

        else
        {
            outputString="Response not as expected\n";
            trackRecordCount=0;
            isError=true;
        }
        return outputString;
    }


    /**
     * Reads bytes of the flash memory.
     * After action isError contains whether the action succeeded or not.
     * @param readPosition Position in memory where to start reading
     * @param size Number of bytes to read
     */
    private void commandReadFlash(int readPosition, int size)
    {
        int         bytesRead;
        int         dataLength;

        DebugLogger.info("* Read flash at 0x"+Integer.toHexString(readPosition)+
                         ", 0x"+Integer.toHexString(size)+" bytes");


        dataLength=connection.commandReadFlash(readPosition, size);

        if (dataLength==size)
        {
            bytesRead=responseHeader[1]*256+responseHeader[2];
            isError=false;
        }
        else
        {
            isError=true;
            DebugLogger.error("Flash read error.");
        }
    }

    /**
     * This command writes data to the flash at given position.
     * The block is subdivided in blocks of WRITEBLOCK_SIZE (0x100) bytes.
     * Each block is send as 0x10 byte chunks after a write command.
     * Note: the chunck contains 0x1F bytes of data and a CRC
     * @param writePosition Address to write to
     * @param size Block size (multiple of 16)
     */
    private void commandWriteFlash(int writePosition, int size)
    {


        DebugLogger.info("* Writing " + Integer.toHexString(size) +
                         " bytes to flash at "+Integer.toHexString(writePosition));

        isError=false;
        
        isError=connection.commandWriteFlash(writePosition, size, writeData);

    }


    /**
     * Erases a 0x1000 block of flash memory
     * After action isError contains whether the action succeeded or not.
     * @param erasePosition Position in memory where to start erasing
     */
   private void commandEraseFlash(int erasePosition)
    {
        String      outputString;
        
        
        DebugLogger.info("* Erasing address: 0x" + Integer.toHexString(erasePosition));
        
        isError=false;
        
        isError=connection.commandEraseFlash(erasePosition);
        
        if (!isError)
        {
            outputString="Block erased at 0x"+Integer.toHexString(erasePosition);
            DebugLogger.info(outputString);
        }
        else
        {
            outputString="Error erasing block at 0x"+Integer.toHexString(erasePosition);
            DebugLogger.error(outputString);
        }

    }

    /**
     * This method returs whether indicated flash eprom block is erased or 
     * not.
     * Basically it reads the 1st 0x10 bytes of the block and compares
     * the 1st 8 bytes against the known sequences that appears to be in
     * erased blocks.
     * After the call isError reflects whether the operation was successful
     * @param block The absolute block number (0x001 - 0x6ff)
     * @return
     */
    boolean isErasedBlock(int block)
    {
        int     firstWord;
        int     secondWord;
        boolean isErased;
        int     firstBlock;
        int     lastBlock;
        int     startAddress;

        DebugLogger.info("Checking erased state of block : 0x" + Integer.toHexString(block));

        isErased=true;

        startAddress    =memMap.getFlashStartAddress();
        firstBlock      =memMap.getTracksFirstPage();
        lastBlock       =memMap.getTracksLastPage();
        
        if (block>=firstBlock && block <=lastBlock)
        {
            commandReadFlash(startAddress+block*Connection.BLOCK_SIZE, 0x10);

            firstWord=((responseData[0]&0xff)<<24) |
                      ((responseData[1]&0xff)<<16) |
                      ((responseData[2]&0xff)<< 8) |
                      ((responseData[3]&0xff)    );
            secondWord=((responseData[4]&0xff)<<24) |
                       ((responseData[5]&0xff)<<16) |
                       ((responseData[6]&0xff)<< 8) |
                       ((responseData[7]&0xff)    );

            isErased=false;

            // Generic sequence GT-120 (in fact never occurs in the GT-800++...)
            if (((firstWord==0xffffffff) && (secondWord==0xffffffff)))
            {
                // Found something else: from this block downwards: purge!!
                isErased=true;
            }            
            
            // Sequences of the GT-800 and GT-800Pro
            if (((firstWord==0xa62d0f21) && (secondWord==0xaffb0f12)) ||
                ((firstWord==0x8dad34a1) && (secondWord==0x962d0ee0)) ||
                ((firstWord==0xbc7b97b3) && (secondWord==0xfacc3c12)) ||
                ((firstWord==0xbd3b69d3) && (secondWord==0xdf8b23e0)))
            {
                // Found something else: from this block downwards: purge!!
                isErased=true;
            }
            
            // Sequences of the GT-820Pro and GT-900Pro
            if (((firstWord==0x151ec766) && (secondWord==0x5d196cca)) ||
                ((firstWord==0x1dffceac) && (secondWord==0x5d196cfa)) ||
                ((firstWord==0x1dffcebc) && (secondWord==0x5d196cea)) ||
                ((firstWord==0x431eb445) && (secondWord==0xe5fc39e0)) ||
                ((firstWord==0x519eae07) && (secondWord==0x5d196cca)) ||
                ((firstWord==0xc7022a11) && (secondWord==0xc7860016)) ||
                ((firstWord==0xc7022a11) && (secondWord==0xc786001e)) ||
                ((firstWord==0x2e9aa522) && (secondWord==0xc7860006)) ||
                ((firstWord==0x2e9aa522) && (secondWord==0xc786001e)) ||
                ((firstWord==0x2e9aa522) && (secondWord==0xc7860016)) ||
                ((firstWord==0x2e9aa522) && (secondWord==0x0811def1)) ||
                ((firstWord==0x56834e90) && (secondWord==0xe57a4d60)) ||
                ((firstWord==0x56834e98) && (secondWord==0xe57a4d60)) ||
                ((firstWord==0x56834e80) && (secondWord==0xe57a4d60)) ||
                ((firstWord==0x5403b241) && (secondWord==0xe57a4d60)) ||
                ((firstWord==0xf69bc843) && (secondWord==0xe5fc39e0)))
            {
                isErased=true;    
            }
            
        }
        else
        {
            DebugLogger.error("Trying to check erased state of illegal block: " + block);
        }

        return isErased;

    }

    /**
     * This method returns whether the device is a GT-800, GT-800Pro,
     * GT-820Pro or GT-900Pro
     * @return True if the device is a GT-800, GT-800Pro,
     * GT-820Pro or GT-900Pro, false if not. Only contains valid 
     * value if isError=true
     */
/*    
    boolean isGT800OrPro()
    {
        boolean isGT800OrPro;

        isGT800OrPro=false;
        
        this.getDeviceType(null);

        if (!isError && 
            ((deviceType==DeviceType.DEVICETYPE_GT800)||
             (deviceType==DeviceType.DEVICETYPE_GT800PRO) ||
             (deviceType==DeviceType.DEVICETYPE_GT820PRO) ||
             (deviceType==DeviceType.DEVICETYPE_GT900PRO)))
        {
            isGT800OrPro=true;
        }

        return isGT800OrPro;
    }
*/

    /**
     * This method compares the response data with the write block
     * @param size Size of the data to compare
     * @return True if blocks are equal, false if not
     */
    private boolean verifyBlock(int size)
    {
        int     i;
        boolean ok;

        ok=true;

        i=0;
        while ((i<size) && ok)
        {
            if (writeData[i]!=responseData[i])
            {
                ok=false;
            }
            i++;
        }

        if(ok)
        {
            DebugLogger.info("Verification OK");
        }
        else
        {
            DebugLogger.error("Verification failed");
        }
        return ok;
    }

    /**
     * This method writes a block and verifies it by reading it back and comparing
     * @param address The address in the Device to write to
     * @param size The size of the data (multiple of 0x100)
     */
    private void writeBlock(int address, int size)
    {
        int         writeRetryCount;
        boolean     verificationOk;

        writeRetryCount=0;
        verificationOk=false;
        while (!verificationOk && writeRetryCount<3)
        {
            commandWriteFlash(address, size);
            commandReadFlash(address, size);
            verificationOk=this.verifyBlock(Connection.BLOCK_SIZE);
            if (!verificationOk)
            {
                writeRetryCount++;
            }
        }
        if (!verificationOk)
        {
            isError=true;
        }
    }

    /**
     * This method reads the settings block from the Device.
     */
    private void getSettingsFromDevice()
    {
        this.commandReadFlash(memMap.getSettingsStartAddress(), Connection.BLOCK_SIZE);
        if (!isError)
        {
            if ((this.deviceType==DeviceType.DEVICETYPE_GT800)    ||
                (this.deviceType==DeviceType.DEVICETYPE_GT800PRO) ||
                (this.deviceType==DeviceType.DEVICETYPE_GT820PRO) ||
                (this.deviceType==DeviceType.DEVICETYPE_GT900PRO))
            {
                logInterval=(responseData[0x04]&0xFF)+
                            ((responseData[0x05]&0xFF)<<8);
            }
            else
            {
                logInterval=-1;
            }
        }
    }
    
    /*************************************************************************\
     * High level communication functions
    \*************************************************************************/

    /**
     * This method tries to find the device type. Based on the model
     * returned by the commandGetModel() the basic type is retrieved:
     * GT-100, GT-120, GT-200 or GT-800(Pro).
     * Based on the model alone it is not possible to discern GT-800 and
     * GT-800Pro. Therefore the assumption is that the major software version
     * for GT-800    is  6 (e.g. 6.11) 
     * for GT-800Pro is  7 (e.g. 7.11).
     * for GT-820    is 12
     * for GT-820Pro is 11
     * for GT-900    is 10
     * for GT-900Pro is  9
     * It is an assumption. There may be other ways...
     * 
     * As a side effect the MemoryMap memMap is set to for the device.
     * This function must be called prior to other functions
     * 
     * @param listener Listener to which progress (0-99) is reported.
     *                 It is allowed to pass null here.
     * @return String describing the result.
     */
    public String getDeviceType(ProgressListener listener)
    {
        String outputString;

        this.commandNmeaSwitch((byte)0x03);
        
        if (listener!=null)
        {
            listener.reportProgress(25);
        }
        
        commandGetModel();
        
        if (listener!=null)
        {
            listener.reportProgress(50);
        }

        if (!isError)
        {
            commandGetIdentification();
        }

        if (listener!=null)
        {
            listener.reportProgress(75);
        }

        if (!isError)
        {
            // Use the software major version to establish the exact device
            // type in case of hte GT800/820/900
            if (modelType==ModelType.MODELTYPE_GT800PLUS)
            {
                switch (this.softwareVersionMajor)
                {
                    case 6:
                        this.deviceType=DeviceType.DEVICETYPE_GT800;
                        this.memMap=new MemoryMapDefault();
                        break;
                    case 7:
                        this.deviceType=DeviceType.DEVICETYPE_GT800PRO;
                        this.memMap=new MemoryMapDefault();
                        break;
                    case 10:
                        this.deviceType=DeviceType.DEVICETYPE_GT900;
                        this.memMap=new MemoryMapDefault();
                        break;
                    case 9:
                        this.deviceType=DeviceType.DEVICETYPE_GT900PRO;
                        this.memMap=new MemoryMapDefault();
                        break;
                    case 12:
                        this.deviceType=DeviceType.DEVICETYPE_GT820;
                        this.memMap=new MemoryMapDefault();
                        break;
                    case 11:
                        this.deviceType=DeviceType.DEVICETYPE_GT820PRO;
                        this.memMap=new MemoryMapDefault();
                        break;
                }
            }
            else if (modelType==ModelType.MODELTYPE_GT120)
            {
                this.deviceType=DeviceType.DEVICETYPE_GT120;
                        this.memMap=new MemoryMapGT120();
            }
            outputString="Device type established: "+getDeviceTypeDescription(deviceType)+"\n";
        }
        else
        {
            outputString="Error establishing device type";
        }

        if (listener!=null)
        {
            listener.reportProgress(99);
        }


        return outputString;
    }


    /**
     * This method executes a full download, including pre-processing
     * of the downloaded data (i.e. subdivision in tracks and segments).
     * If caching is enabled it tries to fetch from cache. If not in cache,
     * it updates the cache. The cache represents the flash memory containing
     * the tracks.
     * Note: progress is reported from 0-99. It should not report 100. 
     * @param listener Listener to which progress (0-99) is reported
     * @return String describing the result of the download
     */
    String downloadTracks(ProgressListener listener)
    {
        int         block;                  //
        int         recordsLeft;            //
        TrackLog    trackLog;               // The log containing track records
        DeviceLog   deviceLog;              // The log containing device log records
        String      outputString;           // The result of the fuction to be displayed
        int         recordsPerBlock;        // number of track records in one block
        int         numberOfBlocks;         // number of blocks containing 'new' records
        int         mostRecentBlock;        // This is the most recently written block in track memory
        int         firstEmptyBlock;        // absolute flash block number in track log that should be empty
        int         oldestPage;             // block on the device containing the oldest data
        int         firstPage;              // first block (absolute) of the track log in flash
        int         lastPage;               // last block (absolute) of the track log
        int         blocksToRead;           // total number of blocks to read
        int         totalBlockCount;        // total blocks read
        boolean     doInitCache;            // Indicates whether the cache must be initialised 
        
        DebugLogger.info("* Downloading tracks");

        
        // Point of starting: no error
        // The variable is set by the communication functions
        this.isError        =false;

        // Retrieve the type of the device.
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);
        
        // Some info on the track log in flash
        recordsPerBlock     =Connection.BLOCK_SIZE/TrackLogPoint.TRACKLOGPOINT_RECORDSIZE;
        firstPage           =memMap.getTracksFirstPage();
        lastPage            =memMap.getTracksLastPage();
        

        // If the caching mechanism is required, intitialise it.
        // This reads the cache from disk (if present)
        if (isCaching) 
        {
            doInitCache=false;
            // Check if the cache has not been initialised before
            if (cache==null)
            {
                doInitCache=true;
            }
            else
            {
                // If it has been intialised before, check if it belongs to this device
                // I.e. check if not another device has been attached since previous download
                if (!cache.checkCache(getDeviceTypeDescription(this.deviceType), this.serialNumber))
                {
                    doInitCache=true;
                }
            }
            
            if (doInitCache)
            {
                cache=new TrackMemoryCache(firstPage,
                                           lastPage,
                                           Connection.BLOCK_SIZE,
                                           getDeviceTypeDescription(this.deviceType), 
                                           this.serialNumber);
            }
        }
        
        
        
        // If the entire track log is empty, things go wrong
        // since commandGetCount() returns a number of track log points
        // which do not exist.
        // Therefore check if the 1st block is erased.
        if (!isErasedBlock(firstPage) && !isError)
        {
            // Device track log makes sense. We are going to read it.
            
            // The track record log
            trackLog =TrackLog.getInstance();
            deviceLog=DeviceLog.getInstance();

            // Erase any existing data
            trackLog.initialiseTracklog();
            deviceLog.clear();

            // Set the type of the device that logged the track
            trackLog.setDeviceType(this.deviceType);

            // Get the number of track records from the device. This is
            // the number of records counting from the start of the track log
            // memory. It does not include the records that are left at the 
            // end of the memory when track logging has looped around!
            commandGetCount();

                
            // The nmber total downloaded blocks
            totalBlockCount     =0;


            // Now we calculate the first memory block (of 0x1000 bytes) that
            // should be entirely empty (when not wrapped around).
            if (this.trackRecordCount%recordsPerBlock==0)
            {
                // Track records exactly fit in an integer number of blocks
                numberOfBlocks=this.trackRecordCount/recordsPerBlock;  

            }
            else
            {
                // Track records do not exactly fit an integer number of blocks.
                // Since the integer division is rounded down, we have to add 1
                numberOfBlocks=this.trackRecordCount/recordsPerBlock+1;  
            }

            
            // Block that contains the most recent track data.
            mostRecentBlock=firstPage+numberOfBlocks-1;
            
            // The first empty block (if not wrapped around. Might be the 1st
            // page outside the track memory if track memory is filled up to
            // and including (part of) the lastPage!!)
            firstEmptyBlock=firstPage+numberOfBlocks;
            
            
            // Now we check if the block is really empty. If not, the 
            // track memory has wrapped around and the remainder of the memory
            // contains previous track records from before the wrap around.
            if ((firstEmptyBlock<=lastPage) && (!isErasedBlock(firstEmptyBlock)) && !isError)
            {
                // We now have to download the entire memory....
                blocksToRead=memMap.getTracksPages();
                
                // Now we start reading, validate the cache
                if (isCaching)
                {
                    // check the situation where the track mem is filled up to
                    // and including the last page. In fact this cannot happen
                    // since it was excluded by the previous if...
                    if (firstEmptyBlock>lastPage)
                    {
                        oldestPage=firstPage;
                        DebugLogger.error("Error validating cache. This should not happen");
                    }
                    else
                    {
                        oldestPage=firstEmptyBlock;
                    }
                    // Read the device flash memory. Sets isError.
                    commandReadFlash(Connection.BLOCK_SIZE*oldestPage, 0x10);
                    cache.validateCache(responseData, oldestPage);
                }
                
                block=firstEmptyBlock;

                DebugLogger.info("Track log wrapped around. Reading flash page 0x" +
                                 Integer.toHexString(block)+" to 0x"+
                                 Integer.toHexString(lastPage));
                
                // We are now going to read the oldest data, in the remainder
                // of the flash memory
                while ((block<=lastPage) && !isError)
                {
                    // Read the block. If using cache, read it from cache,
                    // if not, read it from flash.
                    if (isCaching)
                    {
                        // Try to read from cache. If not succesful read from
                        // flash and write to cache
                        if (!cache.getBlock(responseData, block, mostRecentBlock))
                        {
                            // Read the device flash memory. Sets isError.
                            commandReadFlash(Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                            if (!isError)
                            {
                                cache.writeBlock(block, responseData);
                            }
                        }
                    }
                    else
                    {
                        // Read the device flash memory. Sets isError.
                        commandReadFlash(Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                    }

                    // If all went right, we have got one page or block of flash data
                    // We are going to parse it for track or log records
                    if (!isError)
                    {
                        // Process the records
                        // Parse the data and update the trackLog
                        trackLog.appendData(responseData, Connection.BLOCK_SIZE);
                        // Parse the data once more and update the DeviceLog
                        deviceLog.appendData(responseData, Connection.BLOCK_SIZE);
                        listener.reportProgress(99*totalBlockCount/blocksToRead);

                        block++;
                        totalBlockCount++;
                    }
                    else
                    {

                    }
                }
            }
            else
            {
                // Memory has not wrapped around.
                // We only have to read firstPage up to and NOT including 
                // firstEmptyBlock
                
                blocksToRead=numberOfBlocks;

                // Now we start reading, validate the cache.
                // The page containg the oldest data is the 1st page
                // Compare if that corresponds to the page stored in cache
                if (isCaching)
                {
                    oldestPage=firstPage;
                    // Read the device flash memory. Sets isError.
                    commandReadFlash(Connection.BLOCK_SIZE*oldestPage, 0x10);
                    cache.validateCache(responseData, oldestPage);
                }
            
            }


            // We are now going to read the newest data. Right from the start
            // of the flash track memory
            // Read the newest records in batches of 0x1000 records
            block               =firstPage;      // First page of the track data
            recordsLeft         =this.trackRecordCount;
            DebugLogger.info("Reading flash page 0x" +Integer.toHexString(block)+
                             " to 0x"+Integer.toHexString(firstEmptyBlock-1));

            while ((recordsLeft>0) && !isError)
            {

                // Read the block. If using cache, read it from cache,
                // if not, read it from flash.
                if (isCaching)
                {
                    // Try to read from cache. If not succesful read from
                    // flash and write to cache
                    // Note: the last block is always read from flash since 
                    // records could have been added since last time!!
                    if (!cache.getBlock(responseData, block, mostRecentBlock) /*|| (block==firstEmptyBlock-1)*/)
                    {
                        // Read the device flash memory. Sets isError.
                        commandReadFlash(Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                        if (!isError)
                        {
                            cache.writeBlock(block, responseData);
                        }
                    }
                }
                else
                {                
                    // Read the device flash memory. Sets isError.
                    commandReadFlash(Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                }

                // If all went right, we have got one page or block of flash data
                // We are going to parse it for track or log records
                if (!isError)
                {
                    // Process the records
                    if (recordsLeft>recordsPerBlock)
                    {
                        // Parse the data and update the trackLog
                        trackLog.appendData(responseData, Connection.BLOCK_SIZE);
                        // Parse the data once more and update the DeviceLog
                        deviceLog.appendData(responseData, Connection.BLOCK_SIZE);
                        recordsLeft-=recordsPerBlock;
                    }
                    else
                    {
                        // Parse the data and update the trackLog
                        trackLog.appendData(responseData, recordsLeft*TrackLogPoint.TRACKLOGPOINT_RECORDSIZE);
                        // Parse the data once more and update the DeviceLog
                        deviceLog.appendData(responseData, recordsLeft*TrackLogPoint.TRACKLOGPOINT_RECORDSIZE);
                        recordsLeft=0x0000;
                    }

                    listener.reportProgress(99*totalBlockCount/blocksToRead);
                    
                    block++;
                    totalBlockCount++;
                }
                else
                {

                }
            }
            if (block!=firstEmptyBlock)
            {
                DebugLogger.error("Unexpected number of blocks read while reading track log");
            }
            
            if (!isError)
            {
                // Detect the tracks and track segments
                trackLog.finishTracklog();

                // Dump an overview to System.out
                //        log.dumpTracks();

                // Write something about the result of the action
                outputString="Records from device: "+trackRecordCount+
                             " Valid records: "+trackLog.getNumberOfRecords()+
                             " Log records: " + deviceLog.getNumberOfEntries() + "\n";
            }
            else
            {
                outputString="Error while downloading tracks. Please try again\n";
            }
        }
        else
        {
            // Device is empty, so reset cache
            if (isCaching)
            {
                cache.resetCache();
            }
            outputString="The device contains no track log data";
        }

        if (isCaching)
        {
            cache.makeCachePersistent();
        }

        return outputString;
    }

    /**
     * This method reads the waypoint list from the device. The waypoints
     * are stored in 7 blocks. However, not a full number of waypoints
     * fit in one block (block boundary is in the middle of a waypoint).
     * This complicates matters a bit...
     * @param listener Listener to report progress to
     * @return A string describing the result
     */
    String downloadWaypoints(ProgressListener listener)
    {
        int         block;              // block number relative to start of area 
        WaypointLog waypointLog;
        String      outputString;
        boolean     hasMoreData;
        int         remainingData;
        int         i;
        int         next;
        boolean     exit;
        byte[]      record;
        int         waypointsAddress;
        int         waypointsPages;


        DebugLogger.info("* Downloading waypoints");

        // Get the device type. This sets the memmap defining the capabilities
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);
        
        
        // The waypoint record log
        waypointLog=WaypointLog.getInstance();


        // Erase any existing data
        waypointLog.clear();
 
        if (memMap.isWaypointsSupported())
        {

            // The buffer
            record=new byte[Waypoint.WAYPOINT_RECORDSIZE];

            if (!isError)
            {
                waypointLog.setDeviceType(deviceType);
            }

            // Read the records in batches of 0x1000 records
            block           =0;
            isError         =false;
            hasMoreData     =true;
            remainingData   =0;
            exit            =false;
            next            =0;
            waypointsAddress=memMap.getWaypointsStartAddress();
            waypointsPages  =memMap.getWaypointsPages();

            while (hasMoreData && !isError && !exit)
            {
                // This section constrcts the next waypoint record. If 
                // required, a next page of waypoint data is read from flash 
                i=0;
                while (i<Waypoint.WAYPOINT_RECORDSIZE && !isError)
                {
                    if (remainingData<=0)
                    {
                        // More blocks to read?
                        if (block<waypointsPages)
                        {
                            // Read next block
                            commandReadFlash(waypointsAddress+Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                            if (!isError)
                            {
                                remainingData=Connection.BLOCK_SIZE;
                                next=0;
                                block++;
                                listener.reportProgress(99*(block)/waypointsPages);
                            }
                        }
                        else
                        {
                            // No more blocks to read
                            exit=true;
                        }
                    }

                    if (!isError && !exit)
                    {
                        record[i]=responseData[next];
                        next++;
                        remainingData--;
                    }
                    i++;
                }

                // If successfully constructed a record, add it to the log
                // The return value indicates whether this is the last waypoint.
                if (!isError && !exit)
                {
                    hasMoreData=waypointLog.appendData(record, Waypoint.WAYPOINT_RECORDSIZE);
                }
            }

            if (!isError)
            {
                // Write something about the result of the action
                outputString="Waypoints read from device: "+waypointLog.getNumberOfEntries()+"\n";
            }
            else
            {
                outputString="Error while downloading waypoints. Please try again\n";
            }

            waypointLog.dumpLog();
        }
        else
        {
            outputString="Waypoint log not supported\n";
        }
        return outputString;

    }


    /**
     * Erases the track log
     * @param listener mode 00 - dongle mode 01 - tracer mode 03 - configure mode
     * @author Jorgen van der Velde
     */
    String eraseTracks(ProgressListener listener)
    {
        String  outputString;
        int     block;
        boolean purge;
        int     count;
        int     firstWord;
        int     secondWord;
        int     tracksFirstPage;
        int     tracksLastPage;

        DebugLogger.info("* Erasing tracks");
        
        
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);
        
        
        // Check on GT-800, GT-800Pro, GT-820Pro, GT-900Pro.
        // Other devices have smaller memory. Hence other mem map.
        // I did not test this.
        if (memMap.isValidMemoryMap())
        {
            tracksFirstPage     =memMap.getTracksFirstPage();
            tracksLastPage      =memMap.getTracksLastPage();
            block               =tracksLastPage;
            purge               =false;
            isError             =false;
            count               =0;

            // TO DO: the scanning for the 1st non erased block can be more efficient :-)
            
            // Parse the flash eprom blocks from last (0x6ff) to first (0x001)
            // Each block is 0x1000 bytes. So address from 0x6ff000 to 0x001000
            while (block>=tracksFirstPage && !isError)
            {
                // Find the 1st non-erased block (=block with highest address)
                if (!purge)
                {
                    if (!isErasedBlock(block))
                    {
                        purge=true;
                    }
                }


                if (purge && !isError)
                {
                    // If purge and no errors so far: erase the block
                    commandEraseFlash(block*Connection.BLOCK_SIZE);
                    count++;
                }

                // report some progress
                listener.reportProgress(99*(tracksLastPage-block)/tracksLastPage);

                block--;
            }
            if (!isError)
            {
                outputString="Erased "+count+" out of "+
                             (tracksLastPage-tracksFirstPage)+" flash blocks";
            }
            else
            {
                outputString="Error while erasing...";
            }
        }
        else
        {
            outputString="Erasing not supported for non GT-800 or non-Pro devices...";
        }

        return outputString;
    }

    /**
     * This method uploads the route waypoints to the Device.
     * The waypoints are written to a data block of size Connection.BLOCK_SIZE (0x1000).
     * Unfortunately, not a full number of waypoints fit in the datablock.
     * The data block is filled with waypoint data until it is full.
     * The data block it is written to the device. The remainder of the
     * waypoints is written to the next block.
     * If all waypoints are written, the remainder of the memory is filled with
     * 0x00s, until the entire area 0x707000-0x70FFFF is entirely filled.
     * @param listener Listener to report progress to
     * @return String describing the result
     */
    String uploadRoute(ProgressListener listener)
    {
        String                  outputString;
        RouteLog                routeLog;
        RoutePoint              waypoint;
        int                     numberOfWaypoints;
        int                     i;
        int                     j;
        byte[]                  record;
        int                     bytesInBlock;
        ArrayList<RoutePoint>   waypoints;
        int                     block;
        int                     maxWaypoints;
        int                     routeAddress;
        int                     routePages;

        DebugLogger.info("* Uploading route");
        
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);   
        
        if (memMap.isRouteSupported())
        {

            // Clear progress bar
            listener.reportProgress(0);

            isError             =false;

            routeLog            =RouteLog.getInstance();

            numberOfWaypoints   =routeLog.getNumberOfEntries();


            routeAddress        =memMap.getRouteStartAddress();
            routePages          =memMap.getRoutePages();
            maxWaypoints        =memMap.getRouteMaxWaypoints();

            // Check if the waypoints in the route don't exceed the maximum
            if (numberOfWaypoints>maxWaypoints)
            {
                DebugLogger.error("Route contains more waypoints ("+numberOfWaypoints+
                                ") than can be stored on device ("+maxWaypoints+")");
                numberOfWaypoints=maxWaypoints;
            }

            waypoints=routeLog.getWaypoints();

            if (numberOfWaypoints>0)
            {
                i=0;
                bytesInBlock=0;
                block=0;
                while ((i<numberOfWaypoints) && !isError)
                {
                    waypoint=waypoints.get(i);
                    record  =routeLog.getWaypointAsByteArray(i);

                    j=0;
                    while ((j<RoutePoint.ROUTEPOINT_RECORDSIZE) && !isError)
                    {
                        writeData[bytesInBlock]=record[j];
                        bytesInBlock++;

                        if (bytesInBlock==Connection.BLOCK_SIZE)
                        {
                            writeBlock(routeAddress+block*Connection.BLOCK_SIZE, Connection.BLOCK_SIZE);
                            block++;
                            bytesInBlock=0;
                            listener.reportProgress(99*block/routePages);
                        }

                        j++;
                    }
                    i++;
                }

                // If there are bytes remaining, fill the rest of the block
                // with 0x00 and write it
                if (bytesInBlock>0 && !isError)
                {
                    while (bytesInBlock<Connection.BLOCK_SIZE)
                    {
                        writeData[bytesInBlock]=(byte)0x00;
                        bytesInBlock++;
                    }
                    writeBlock(routeAddress+block*Connection.BLOCK_SIZE, Connection.BLOCK_SIZE);
                    block++;
                    listener.reportProgress(99*block/routePages);
                }

                // Fill the remaining EPROM blocks with 0x00
                // Fill the block with 0x00
                bytesInBlock=0;
                while (bytesInBlock<Connection.BLOCK_SIZE)
                {
                    writeData[bytesInBlock]=(byte)0x00;
                    bytesInBlock++;
                }

                // write it as many times as needed
                while (block<routePages && !isError)
                {
                    writeBlock(routeAddress+block*Connection.BLOCK_SIZE, Connection.BLOCK_SIZE);
                    block++;
                    listener.reportProgress(99*block/routePages);
                }

                if (!isError)
                {
                    outputString="Upload of "+numberOfWaypoints+" waypoints succeeded";
                }
                else
                {
                    outputString="Error writing data to device";
                }
            }
            else
            {
                outputString="No waypoints to upload";
            }
        }
        else
        {
            outputString="Device does not support uploadable routes";
        }   
        return outputString;
    }

    
    /**
     * This method uploads the route waypoints to the Device.
     * The waypoints are written to a data block of size Connection.BLOCK_SIZE (0x1000).
     * Unfortunately, not a full number of waypoints fit in the datablock.
     * The data block is filled with waypoint data until it is full.
     * The data block it is written to the device. The remainder of the
     * waypoints is written to the next block.
     * If all waypoints are written, the remainder of the memory is filled with
     * 0x00s, until the entire area 0x707000-0x70FFFF is entirely filled.
     * @param listener Listener to report progress to
     * @return String describing the result
     */
    String eraseRoute(ProgressListener listener)
    {
        String                  outputString;
        RouteLog                routeLog;
        RoutePoint              waypoint;
        int                     numberOfWaypoints;
        int                     i;
        int                     j;
        byte[]                  record;
        int                     bytesInBlock;
        ArrayList<RoutePoint>   waypoints;
        int                     block;
        int                     maxWaypoints;
        int                     routeAddress;
        int                     routePages;

        DebugLogger.info("* Erase route");

        // Clear progress bar
        listener.reportProgress(0);

        isError             =false;

        // Get the device type from the device. MemMap is set corresponding
        // to the device
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);  
        
        if (memMap.isRouteSupported())
        {

            routeAddress        =memMap.getRouteStartAddress();
            routePages          =memMap.getRoutePages();
            maxWaypoints        =memMap.getRouteMaxWaypoints();


            // File the writeblock with 0x00
            bytesInBlock=0;
            while (bytesInBlock<Connection.BLOCK_SIZE)
            {
                writeData[bytesInBlock]=(byte)0x00;
                bytesInBlock++;
            }

            block=0;
            while (block<routePages && !isError)
            {
                writeBlock(routeAddress+block*Connection.BLOCK_SIZE, Connection.BLOCK_SIZE);
                listener.reportProgress(99*block/routePages);            
                block++;
            }

            if (!isError)
            {
                outputString="Uploadable route area cleared";
            }
            else
            {
                outputString="Error writing data to device";
            }
        }
        else
        {
            outputString="Device does not support uploadable routes. Nothing to erase";
        }
        return outputString;
    }
    
    
    
    /**
     * This method downloads the route from the Device.
     * @param listener Listener to report progress to
     * @return String describing the result
     */
    String downloadRoute(ProgressListener listener)
    {
        int         block;
        RouteLog    routeLog;
        String      outputString;
        boolean     hasMoreData;
        int         remainingData;
        int         i;
        int         next;
        boolean     exit;
        byte[]      record;
        int         routeAddress;
        int         routePages;        


        DebugLogger.info("* Downloading route");

        // Get the device type from the device. MemMap is set corresponding
        // to the device
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);         
        
        if (memMap.isRouteSupported())
        {
            routeAddress        =memMap.getRouteStartAddress();
            routePages          =memMap.getRoutePages();


            // The buffer
            record=new byte[RoutePoint.ROUTEPOINT_RECORDSIZE];

            // The waypoint record log
            routeLog=RouteLog.getInstance();


            // Erase any existing data
            routeLog.clear();

            // Pass null here as listener, otherwise the progress bar
            // goes berzerk
            this.getDeviceType(null);

            if (!isError)
            {
                routeLog.setDeviceType(deviceType);
            }

        // Read the records in batches of 0x1000 records
            block           =0;
            isError         =false;
            hasMoreData     =true;
            remainingData   =0;
            exit            =false;
            next            =0;

            while (hasMoreData && !isError && !exit)
            {
                // Construct the record
                i=0;
                while (i<RoutePoint.ROUTEPOINT_RECORDSIZE && !isError)
                {
                    if (remainingData<=0)
                    {
                        // More blocks to read?
                        if (block<routePages)
                        {
                            // Read next block
                            commandReadFlash(routeAddress+Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                            if (!isError)
                            {
                                remainingData=Connection.BLOCK_SIZE;
                                next=0;
                                block++;
                                listener.reportProgress(99*(block)/routePages);
                            }
                        }
                        else
                        {
                            // No more blocks to read
                            exit=true;
                        }
                    }

                    if (!isError && !exit)
                    {
                        record[i]=responseData[next];
                        next++;
                        remainingData--;
                    }
                    i++;
                }

                // If successfully constructed a record, add it to the log
                if (!isError && !exit)
                {
                    hasMoreData=routeLog.appendData(record, RoutePoint.ROUTEPOINT_RECORDSIZE);
                }
            }

            if (!isError)
            {
                // Write something about the result of the action
                outputString="Route waypoints read from device: "+routeLog.getNumberOfEntries()+"\n";
            }
            else
            {
                outputString="Error while downloading tracks. Please try again\n";
            }

            routeLog.dumpLog();
        }
        else
        {
            outputString="Device does not support uploadable routes. Nothing to download";
        }
        return outputString;

    }



    /**
     * This method translates the device type passed to a string representation.
     * @param deviceType The device type to translate
     * @return The device type as string
     */
    public static String getDeviceTypeDescription(DeviceType deviceType)
    {
        String outputString;

        switch (deviceType)
        {
            case DEVICETYPE_GT100:
                outputString="GT-100";
                break;
            case DEVICETYPE_GT120:
                outputString="GT-120";
                break;
            case DEVICETYPE_GT200:
                outputString="GT-200";
                break;
            case DEVICETYPE_GT800:
                outputString="GT-800";
                break;
            case DEVICETYPE_GT800PRO:
                outputString="GT-800PRO";
                break;
            case DEVICETYPE_GT820:
                outputString="GT-820";
                break;
            case DEVICETYPE_GT820PRO:
                outputString="GT-820PRO";
                break;
            case DEVICETYPE_GT900:
                outputString="GT-900";
                break;
            case DEVICETYPE_GT900PRO:
                outputString="GT-900PRO";
                break;
            default:
                outputString="Unknown";
                break;
        }

        return outputString;
    }

    /**
     * This method retrieves info from the device
     * @param listener Listener to report progress to
     * @return String containing info.
     */
    public String getInfo(ProgressListener listener)
    {
        String outputString;
        String resultString;

        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);        
        
        getSettingsFromDevice();
        listener.reportProgress(25);        
        outputString=commandGetIdentification();
        listener.reportProgress(50);
        resultString=commandGetModel();
        outputString+=resultString;
        listener.reportProgress(75);
        resultString=commandGetCount();
        outputString+=resultString;
        return outputString;
    }
    
    
    /**
     * For debugging: scan of first 16 bytes of each memory block
     */
    public void scanFlash()
    {
        int i;
        int j;
        String fileName;
        FileWriter fileWriter;            
        BufferedWriter bufferedWriter;  
        int flashStartAddress;
        int flashPages;

        isError=false;
        
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);  
        
        fileName="flashscan_"+getDeviceTypeDescription(deviceType);
        fileName+="_"+serialNumber+".txt";
        
        DebugLogger.info("Writing to file "+fileName);
        
        try
        {
            fileWriter = new FileWriter(fileName);

            // Always wrap FileWriter in BufferedWriter.
            bufferedWriter = new BufferedWriter(fileWriter);
            
            flashStartAddress=memMap.getFlashStartAddress();
            flashPages       =memMap.getFlashPages();
            i=0;
            while (i<flashPages)
            {
                this.commandReadFlash(flashStartAddress+i*Connection.BLOCK_SIZE, 0x10);
                
                bufferedWriter.write(String.format("%06x: ", flashStartAddress+i*Connection.BLOCK_SIZE));

                j=0;
                while (j<8)
                {
                    bufferedWriter.write(String.format("%02x", responseData[j]));
                    j++;
                }
                bufferedWriter.write("\n");
                i++;
            }
            
            bufferedWriter.close();
            fileWriter.close();	            
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing file "+fileName);
        }        
        
    }

    
    /**
     * For debugging: scan of first 16 bytes of each memory block. It makes a
     * list of unique start strings of the page. It appears there is a limited
     * set of start strings when a page is erased (and 0xffffffff is not one of 
     * them)
     */
    public void scanFlashForUniqueStrings()
    {
        int i;
        int j;
        String fileName;
        FileWriter fileWriter;            
        BufferedWriter bufferedWriter;  
        int flashStartAddress;
        int flashPages;
        String flashString;
        String testString;
        boolean found;
        ArrayList<String> strings;
        Iterator<String> iterator;

        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);          
        
        flashStartAddress=memMap.getTracksStartAddress();
        flashPages       =memMap.getTracksPages();        
        
        strings=new ArrayList<String>();
        
        // Find list of unique strings
        i=0;
        while (i<flashPages)
        {
            this.commandReadFlash(flashStartAddress+i*Connection.BLOCK_SIZE, 0x10);

            
            flashString="";
            j=0;
            while (j<8)
            {
                flashString+=String.format("%02x", responseData[j]);
                j++;
            }
            
            
            iterator=strings.iterator();
            found=false;
            while (iterator.hasNext() && !found)
            {
                testString=iterator.next();
                if (flashString.equals(testString))
                {
                    found=true;
                }
            }
            
            if (!found)
            {
                strings.add(flashString);
            }
            

            i++;
        }
       
        isError=false;
        
        getDeviceType(null);
        
        fileName="flashscan_uniquestrings_"+getDeviceTypeDescription(deviceType);
        fileName+="_"+serialNumber+".txt";
        
        DebugLogger.info("Writing to file "+fileName);
        
        try
        {
            fileWriter = new FileWriter(fileName);

            // Always wrap FileWriter in BufferedWriter.
            bufferedWriter = new BufferedWriter(fileWriter);
            
            iterator=strings.iterator();
            while (iterator.hasNext())
            {
                testString=iterator.next();
                bufferedWriter.write(testString+"\n");
            }
            
            bufferedWriter.close();
            fileWriter.close();	            
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing file "+fileName);
        }        
        
    }
    
    
    /**
     * For debugging: write the response arrays
     * @param fileName 
     */
    private void writeResponse(String fileName)
    {
        FileOutputStream outputStream;
        
        try
        {
            outputStream = new FileOutputStream(fileName);

            outputStream.write(this.responseHeader, 0, connection.getResponseHeaderLength());
            outputStream.write(this.responseData, 0, connection.getResponseDataLength());

            outputStream.close();	            
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing response file "+fileName);
        }        
    }
    
    /**
     * For debugging: dump content of flash memory.
     */
    public String simulationDump(String path,  ProgressListener listener)
    {
        int i;
        String fileName;
        FileOutputStream outputStream;
        int flashStartAddress;
        int flashPages;        
            
        
        isError=false;
        
        DebugLogger.info("Writing dump for simulation");

        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);          
        
        // Write memory dump
        fileName=path+"memory_dump.dat";
        try
        {
            outputStream = new FileOutputStream(fileName);
            
            flashStartAddress=memMap.getFlashStartAddress();
            flashPages       =memMap.getFlashPages();
            
            i=0;
            while ((i<flashPages) && !isError)
            {
                this.commandReadFlash(flashStartAddress+i*Connection.BLOCK_SIZE, Connection.BLOCK_SIZE);

                outputStream.write(this.responseData);

                if (listener!=null)
                {
                    listener.reportProgress(95*i/flashPages);
                }
                
                i++;
            }
            outputStream.close();	            
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing file "+fileName);
        }

        this.commandNmeaSwitch((byte)0x03);
        writeResponse(path+"acknowledge_response.dat");
       
        
        this.commandGetModel();
        writeResponse(path+"modelcmd_response.dat");
        
        this.commandGetIdentification();
        writeResponse(path+"identificationcmd_response.dat");
        
        this.commandGetCount();
        writeResponse(path+"countcmd_response.dat");
        
        this.commandNmeaSwitch((byte)0x00);

        if (listener!=null)
        {
            listener.reportProgress(99);
        }       

        return "Simulation data dumped to "+path;
    }
    
    /**
     * For debugging: Save the device settings to a file
     */
    public String saveDeviceSettings(String path,  ProgressListener listener)
    {
        String fileName;
        FileOutputStream outputStream;
        int settingsStartAddress;
            
        
        isError=false;
        
        DebugLogger.info("Writing settings");
        
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);          

        // Write memory dump
        fileName=path+"settings.dat";
        try
        {
            outputStream = new FileOutputStream(fileName);
            
            settingsStartAddress=memMap.getSettingsStartAddress();
            
            this.commandReadFlash(settingsStartAddress, Connection.BLOCK_SIZE);

            outputStream.write(this.responseData);

            outputStream.close();	            
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing file "+fileName);
        }

        if (listener!=null)
        {
            listener.reportProgress(99);
        }       

        return "Device settings written to "+fileName;
    }

    /**
     * For debugging: Restore the device settings to the Device from a file
     */
    public String restoreDeviceSettings(String path,  ProgressListener listener)
    {
        String fileName;
        int settingsStartAddress;
            
        
        isError=false;
        
        DebugLogger.info("Restore device settings to the device");

        // Write memory dump
        fileName=path+"settings.dat";

        return "Not implemented yet";
    }

    /**
     * For debugging: Save the device settings to a file
     * @param path
     * @param listener
     * @return 
     */
    public String saveDeviceSettingsAsText(String path,  ProgressListener listener)
    {
        String          fileName;
        FileWriter      fileWriter;            
        BufferedWriter  bufferedWriter;  
        int             settingsStartAddress;
        int             theInt;
        
        isError=false;
        
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);  
        
        
        fileName=path+"settings_"+getDeviceTypeDescription(deviceType);
        fileName+="_"+serialNumber+".txt";
        
        DebugLogger.info("Writing settings to text file "+fileName);

        settingsStartAddress=memMap.getSettingsStartAddress();

        this.commandReadFlash(settingsStartAddress, Connection.BLOCK_SIZE);

        if (!isError)
        {        
            try
            {
                fileWriter = new FileWriter(fileName);

                // Always wrap FileWriter in BufferedWriter.
                bufferedWriter = new BufferedWriter(fileWriter);

                bufferedWriter.write("SETTINGS "+getDeviceTypeDescription(deviceType)+
                                    " SERIAL: "+serialNumber+"\n\n");                
                
                // Functions for the GT800+
                if (    (this.deviceType==DeviceType.DEVICETYPE_GT800) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT800PRO) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT820) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT820PRO) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT900) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT900PRO))
                {
                    bufferedWriter.write("Log interval           : "+
                                        ToolBox.bytesToIntLe(responseData, 0x0004, 2)+" s\n");

                    if ((responseData[0x0012]&0x10)>0)
                    {
                        bufferedWriter.write("Power saving           : On\n");
                    }            
                    else
                    {
                        bufferedWriter.write("Power saving           : Off\n");
                    }            

                    if ((responseData[0x0013]&0x80)>0)
                    {
                        bufferedWriter.write("Clock 12/24h           : 12h\n");
                    }            
                    else
                    {
                        bufferedWriter.write("Clock 12/24h           : 24h\n");
                    }            

                    if ((responseData[0x0013]&0x10)>0)
                    {
                        bufferedWriter.write("Screen rotation        : Landscape\n");
                    }            
                    else
                    {
                        bufferedWriter.write("Screen rotation        : Portrait\n");
                    }            

                    bufferedWriter.write("Time offset to UTC     : "+
                                        ToolBox.bytesToIntLe(responseData, 0x0070, 2)+" min\n");

                    theInt=ToolBox.bytesToIntLe(responseData, 0x0072, 2);
                    if (theInt>0)
                    {
                        bufferedWriter.write("Autopause speed        : "+ theInt/10 +" km/h\n");
                    }
                    else
                    {
                        bufferedWriter.write("Autopause speed        : Off\n");
                    }

                    if (responseData[0x0074]==0)
                    {
                        bufferedWriter.write("Pedometer sens.        : High\n");
                    }            
                    else if (responseData[0x0074]==1)
                    {
                        bufferedWriter.write("Pedometer sens.        : Normal\n");
                    }            
                    else if (responseData[0x0074]==2)
                    {
                        bufferedWriter.write("Pedometer sens.        : Low\n");
                    }            
                    else
                    {
                        bufferedWriter.write("Pedometer sens.        : Unknown\n");
                    }            

                    if (responseData[0x0075]==0x00)
                    {
                        bufferedWriter.write("Units                  : Metric\n");
                    }            
                    else if (responseData[0x0075]==0x01)
                    {
                        bufferedWriter.write("Units                  : Imperial\n");
                    }            
                    else
                    {
                        bufferedWriter.write("Units                  : Unknown\n");
                    }            

                    if (responseData[0x0076]==0x00)
                    {
                        bufferedWriter.write("Language               : English\n");
                    }            
                    else if (responseData[0x0076]==0x01)
                    {
                        bufferedWriter.write("Language               : Chinese, japanese, whatever\n");
                    }            
                    else if (responseData[0x0076]==0x02)
                    {
                        bufferedWriter.write("Language               : Chinese, japanese, whatever\n");
                    }            
                    else if (responseData[0x0076]==0x03)
                    {
                        bufferedWriter.write("Language               : Chinese, japanese, whatever\n");
                    }            
                    else if (responseData[0x0076]==0x04)
                    {
                        bufferedWriter.write("Language               : German\n");
                    }            
                    else if (responseData[0x0076]==0x05)
                    {
                        bufferedWriter.write("Language               : French\n");
                    }            
                    else if (responseData[0x0076]==0x06)
                    {
                        bufferedWriter.write("Language               : Dutch\n");
                    }            
                    else if (responseData[0x0076]==0x07)
                    {
                        bufferedWriter.write("Language               : Italian\n");
                    }            
                    else if (responseData[0x0076]==0x08)
                    {
                        bufferedWriter.write("Language               : Spanish\n");
                    }            
                    else if (responseData[0x0076]==0x09)
                    {
                        bufferedWriter.write("Language               : Russian\n");
                    }            
                    bufferedWriter.write("Altitude calibration   : "+
                                        ToolBox.bytesToIntLe(responseData, 0x0080, 4)+" cm\n");
                }
                
                // Extended fucntionality for the GT-820 and GT-900
                if (    (this.deviceType==DeviceType.DEVICETYPE_GT820) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT820PRO) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT900) ||
                        (this.deviceType==DeviceType.DEVICETYPE_GT900PRO))
                {
                    bufferedWriter.write("Profile length         : "+
                                         (ToolBox.bytesToIntLe(responseData, 0x00B0, 2)/10)+" cm\n");

                    bufferedWriter.write("Profile weight         : "+
                                         (ToolBox.bytesToIntLe(responseData, 0x00B2, 2)/100)+" kg\n");

                    bufferedWriter.write("Profile age            : "+
                                         responseData[0x00b4]+" years\n");

                    if (responseData[0x00b5]==0x00)
                    {
                        bufferedWriter.write("Gender                 : female\n");
                    }
                    else if (responseData[0x00b5]==0x01)
                    {
                        bufferedWriter.write("Gender                 : male\n");
                    }
                    else
                    {
                        bufferedWriter.write("Gender                 : unknown\n");
                    }

                    if ((responseData[0x0013]&0x40)>0)
                    {
                        bufferedWriter.write("Sound                  : Off\n");
                    }            
                    else
                    {
                        bufferedWriter.write("Sound                  : On\n");
                    }            

                    if (responseData[0x00cc]==0)
                    {
                        bufferedWriter.write("Alert heartrate high   : Off\n");
                    }
                    else if (responseData[0x00cc]==1)
                    {
                        bufferedWriter.write("Alert heartrate high   : >106 (zone 1)\n");
                    }
                    else if (responseData[0x00cc]==2)
                    {
                        bufferedWriter.write("Alert heartrate high   : >123 (zone 2)\n");
                    }
                    else if (responseData[0x00cc]==3)
                    {
                        bufferedWriter.write("Alert heartrate high   : >141 (zone 3)\n");
                    }
                    else if (responseData[0x00cc]==4)
                    {
                        bufferedWriter.write("Alert heartrate high   : >156 (zone 4)\n");
                    }
                    else if (responseData[0x00cc]==5)
                    {
                        bufferedWriter.write("Alert heartrate high   : >177 (zone 5)\n");
                    }
                    else
                    {
                        bufferedWriter.write("Alert heartrate high   : unknonwn\n");
                    }


                    if (responseData[0x00cd]==0)
                    {
                        bufferedWriter.write("Alert heartrate low    : Off\n");
                    }
                    else if (responseData[0x00cd]==1)
                    {
                        bufferedWriter.write("Alert heartrate low    : <88 (zone 1)\n");
                    }
                    else if (responseData[0x00cd]==2)
                    {
                        bufferedWriter.write("Alert heartrate low    : <106 (zone 2)\n");
                    }
                    else if (responseData[0x00cd]==3)
                    {
                        bufferedWriter.write("Alert heartrate low    : <123 (zone 3)\n");
                    }
                    else if (responseData[0x00cd]==4)
                    {
                        bufferedWriter.write("Alert heartrate low    : <141 (zone 4)\n");
                    }
                    else if (responseData[0x00cd]==5)
                    {
                        bufferedWriter.write("Alert heartrate low    : <159 (zone 5)\n");
                    }
                    else
                    {
                        bufferedWriter.write("Alert heartrate low    : unknonwn\n");
                    }

                    theInt=ToolBox.bytesToIntLe(responseData, 0x00C6, 2);
                    if (theInt==0x00)
                    {
                        bufferedWriter.write("Alert duration         : Off\n");
                    }
                    else
                    {
                        bufferedWriter.write("Alert duration         : " + theInt + "  min\n");
                    }

                    theInt=ToolBox.bytesToIntLe(responseData, 0x00D4, 4);
                    if (theInt==0x00)
                    {
                        bufferedWriter.write("Alert distance         : Off\n");
                    }
                    else
                    {
                        bufferedWriter.write("Alert distance         : " + theInt + "  m\n");
                    }
                }          
                
                if (this.deviceType==DeviceType.DEVICETYPE_GT120)
                {
                    bufferedWriter.write("No supported yet");
                }
                
                bufferedWriter.close();
                fileWriter.close();	            
            }
            catch(IOException e)
            {
                DebugLogger.error("Error writing file "+fileName);
            }        
        }
        return "Settings written to "+fileName;
    }

    
    /**
     * This method verifies the cache file by comparing it with the content
     * of the device flash memory. It must be executed after connecting the 
     * device and after performing a full download. After that, the cache 
     * memory should be identical to the device memory.
     * @param listener Listener to which progress (0-99) is reported
     * @return String describing the result of the download
     */
    String verifyCache(ProgressListener listener)
    {
        int         block;                  //
        int         recordsLeft;            //
        String      outputString;           // The result of the fuction to be displayed
        int         recordsPerBlock;        // number of track records in one block
        int         numberOfBlocks;         // number of blocks containing 'new' records
        int         firstEmptyBlock;        // absolute flash block number in track log that should be empty
        int         oldestPage;             // block on the device containing the oldest data
        int         firstPage;              // first block (absolute) of the track log in flash
        int         lastPage;               // last block (absolute) of the track log
        int         blocksToRead;           // total number of blocks to read
        int         totalBlockCount;        // total blocks read
        boolean     cacheIsOk;              // Indicates whether the cache is ok so far 
        boolean     exit;
        
        DebugLogger.info("* Verifying the cache file contents");

        
        // Point of starting: no error
        // The variable is set by the communication functions
        this.isError        =false;
        cacheIsOk           =true;
        exit                =false;
        outputString        ="";

        // Retrieve the type of the device.
        // Pass null here as listener, otherwise the progress bar
        // goes berzerk
        this.getDeviceType(null);
        
        
        // Some info on the track log in flash
        recordsPerBlock     =Connection.BLOCK_SIZE/TrackLogPoint.TRACKLOGPOINT_RECORDSIZE;
        firstPage           =memMap.getTracksFirstPage();
        lastPage            =memMap.getTracksLastPage();
        
        
        if (isError)
        {
            exit=true;
        }

        if (!this.isCaching)
        {
            outputString="Caching is disabled. Enable caching first\n";
            exit        =true;
        }
        else
        {
            // Check if the cache has not been initialised before for the 
            // connected device. It should be, since we assume a track download
            // 
            if (cache==null)
            {
                outputString="Please, perform a track download first\n";
                exit        =true;
            }
            else
            {
                // If it has been intialised before, check if it belongs to this device
                // I.e. check if not another device has been attached since previous download
                if (!cache.checkCache(getDeviceTypeDescription(this.deviceType), this.serialNumber))
                {
                    exit        =true;
                    outputString="Please, perform a track download first\n";
                }
            }
        }
        
        
        // If the entire track log is empty, things go wrong
        // since commandGetCount() returns a number of track log points
        // which do not exist.
        // Therefore check if the 1st block is erased.
        if (!exit)
        {
            if (!isErasedBlock(firstPage))
            {
                // Device track log makes sense. We are going to read it.


                // Get the number of track records from the device. This is
                // the number of records counting from the start of the track log
                // memory. It does not include the records that are left at the 
                // end of the memory when track logging has looped around!
                commandGetCount();


                // The nmber total downloaded blocks
                totalBlockCount     =0;

                // Now we calculate the first memory block (of 0x1000 bytes) that
                // should be entirely empty (when not wrapped around).
                if (this.trackRecordCount%recordsPerBlock==0)
                {
                    // Track records exactly fit in an integer number of blocks
                    numberOfBlocks=this.trackRecordCount/recordsPerBlock;  

                }
                else
                {
                    // Track records do not exactly fit an integer number of blocks.
                    // Since the integer division is rounded down, we have to add 1
                    numberOfBlocks=this.trackRecordCount/recordsPerBlock+1;  
                }


                // The first empty block (if not wrapped around. Might be the 1st
                // page outside the track memory if track memory is filled up to
                // and including (part of) the lastPage!!)
                firstEmptyBlock=firstPage+numberOfBlocks;


                // Now check an important parameter in the cache: the last block that has
                // been written.
                DebugLogger.info("Checking last block written number: 0x"+Integer.toHexString(firstEmptyBlock-1));
                if (!cache.checkLastBlockWritten(firstEmptyBlock-1))
                {
                    cacheIsOk=false;
                    DebugLogger.info("Cache lastBlockWritten parameter value is not ok");
                }
                else
                {
                    DebugLogger.info("Cache lastBlockWritten parameter value is ok");
                }

                // Now we check if the block is really empty. If not, the 
                // track memory has wrapped around and the remainder of the memory
                // contains previous track records from before the wrap around.
                if ((firstEmptyBlock<=lastPage) && (!isErasedBlock(firstEmptyBlock)) && !isError)
                {
                    // We now have to download the entire memory....
                    blocksToRead=memMap.getTracksPages();

                    block=firstEmptyBlock;

                    DebugLogger.info("Track log wrapped around. Checking flash page 0x" +
                                     Integer.toHexString(block)+" to 0x"+
                                     Integer.toHexString(lastPage));

                    // We are now going to read the oldest data, in the remainder
                    // of the flash memory
                    while ((block<=lastPage) && !isError)
                    {
                        commandReadFlash(Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                        if (!cache.checkBlockContent(responseData, block))
                        {
                            DebugLogger.info("Cache page 0x"+Integer.toHexString(block)+" is not ok!");
                            cacheIsOk=false;
                        }
                        block++;
                        totalBlockCount++;
                        listener.reportProgress(99*totalBlockCount/blocksToRead);
                    }
                }
                else
                {
                    // Memory has not wrapped around.
                    // We only have to read firstPage up to and NOT including 
                    // firstEmptyBlock
                    blocksToRead=numberOfBlocks;

                }


                // We are now going to read the newest data. Right from the start
                // of the flash track memory
                // Read the newest records in batches of 0x1000 records
                block               =firstPage;      // First page of the track data
                recordsLeft         =this.trackRecordCount;
                DebugLogger.info("Checking flash page 0x" +Integer.toHexString(block)+
                                 " to 0x"+Integer.toHexString(firstEmptyBlock-1));

                while ((block<firstEmptyBlock) && !isError)
                {
                    commandReadFlash(Connection.BLOCK_SIZE*block, Connection.BLOCK_SIZE);
                    if (!cache.checkBlockContent(responseData, block))
                    {
                        DebugLogger.info("Cache page 0x"+Integer.toHexString(block)+" is not ok!");
                        cacheIsOk=false;
                    }
                    block++;
                    totalBlockCount++;

                    listener.reportProgress(99*totalBlockCount/blocksToRead);

                }

                if (!isError && !exit)
                {
                    if (cacheIsOk)
                    {
                        outputString="Cache file is ok!";
                    }
                    else
                    {
                        outputString="Cache file is not ok!";
                    }
                }

            }
            else
            {
                outputString="The device contains no track log data";
            }
        }
        

        return outputString;
    }
    
    
    
}
