/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

import java.io.FileWriter;
import java.io.PrintWriter;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.ListIterator;
import java.util.Iterator;

/**
 * This class writes Tracks and Waypoints to CSV file
 * @author Jorgen
 */
public class CsvWriter implements IgotuWriter
{
    /** The one and only instance of this class*/
    private static CsvWriter theInstance=null;
    
    private static final String SEPARATOR=";";
    
    private CsvWriter()
    {
        
    }
    
    public static CsvWriter getInstance()
    {
        if (theInstance==null)
        {
            theInstance=new CsvWriter();
        }
        
        return theInstance;
    }
    
    
    /**
     * This method writes device info
     * @param writer Writer to use for writing to
     */
    private void writeDeviceInfo(PrintWriter writer, Device.DeviceType type)
    {
        writer.write("Device"+SEPARATOR+Device.getDeviceTypeDescription(type)+"\n");
        writer.write("\n");
    }
    
    /**
     * This method prints a resumé on the track
     * @param writer Writer to write to
     * @param track Track whose info is displayed
     */
    private void writeTrackInfo(PrintWriter writer, Track track)
    {
        writer.write("Start"+SEPARATOR+track.getLocalStartTime().format("YYYY-MM-DD hh:mm:ss")+"\n");
        writer.write("End"+SEPARATOR+track.getLocalEndTime().format("YYYY-MM-DD hh:mm:ss")+"\n");
        writer.write("Average heart rate (bpm)"+SEPARATOR+track.getAverageHeartRate()+"\n");
        writer.write("Maximum heart rate (bpm)"+SEPARATOR+track.getMaxHeartRate()+"\n");
        writer.write("Calories estimation (kCal)"+SEPARATOR+track.getCalories()+"\n");
        writer.write("Segments"+SEPARATOR+track.getNumberOfSegments()+"\n");
        writer.write("Trackpoints"+SEPARATOR+track.getNumberOfTrackPoints()+"\n");
        writer.write("Waypoints"+SEPARATOR+track.getNumberOfWaypoints()+"\n");
        writer.write("Heartrate points"+SEPARATOR+track.getNumberOfHeartRatePoints()+"\n");
        writer.write("\n");
    }
    
    /**
     * This method writes the track header line
     * @param writer Writer to write to
     */
    private void writeTrackHeader(PrintWriter writer)
    {
            // Date-time stamp
            writer.print("Date-time");
            writer.print(SEPARATOR);

            // Latitude
            writer.print("Latitude (deg)");
            writer.print(SEPARATOR);

            // Longitude
            writer.print("Longitude (deg)");
            writer.print(SEPARATOR);

            // Elevation - by atmospheric pressure
            writer.print("Barometric height (m)");
            writer.print(SEPARATOR);

            // Elevation - by gps
            writer.print("GPS heigth (m)");
            writer.print(SEPARATOR);

            // Speed in m/s
            writer.print("Speed (m/s)");
            writer.print(SEPARATOR);

            // Heading in degrees
            writer.print("Heading (deg)");
            writer.print(SEPARATOR);

            // Ehpe
            writer.print("Precision (m)");
            writer.print(SEPARATOR);

            // Satellite number
            writer.print("Number of satellites");
            writer.print(SEPARATOR);

            // Heart rate
            writer.print("Heart rate (bpm)");

            writer.print("\n");
        
    }
    
    private void writeTrackPoint(PrintWriter writer, TrackLogPoint point)
    {
        // Date-time stamp
        writer.print(point.getDateTime().format("YYYY-MM-DD hh:mm:ss"));
        writer.print(SEPARATOR);

        // Latitude
        writer.print(Double.toString(point.getLatitude()));
        writer.print(SEPARATOR);

        // Longitude
        writer.print(Double.toString(point.getLongitude()));
        writer.print(SEPARATOR);

        // Elevation - by atmospheric pressure
        writer.print(Double.toString(point.getElevationByAtmosphericPresure()));
        writer.print(SEPARATOR);

        // Elevation - by gps
        writer.print(Double.toString(point.getElevationByGps()));
        writer.print(SEPARATOR);

        // Speed in m/s
        writer.print(Double.toString(point.getSpeed()));
        writer.print(SEPARATOR);

        // Heading in degrees
        writer.print(Double.toString(point.getCourse()));
        writer.print(SEPARATOR);

        // Ehpe
        writer.print(point.getEhpe());
        writer.print(SEPARATOR);

        // Satellite number
        writer.print(point.getSatelliteNumber());
        writer.print(SEPARATOR);

        // Heart rate
        writer.print(point.getHeartRate());

        writer.print("\n");
        
    }
    
    
    /**
     * This method writes the indicated track as stored in the TrackLog to file
     * @param fileName File name
     * @param trackNo Number of the track in the TrackLog
     * @param trackName Name of the track, can be used for identification
     */
    public void writeTrackToFile(String fileName, int trackNo, String trackName)
    {
        TrackLog                        log;
        Track                           track;
        ArrayList<TrackSegment>         segments;
        TrackSegment                    segment;
        ArrayList<TrackLogPoint>        points;
        TrackLogPoint                   point;
        Iterator<TrackSegment>          segmentIterator;
        Iterator<TrackLogPoint>         pointIterator;
        
        
        log=TrackLog.getInstance();
        track=log.getTrack(trackNo);

        

        try
        {
            FileWriter fw = new FileWriter(fileName);
            PrintWriter pw = new PrintWriter(fw);

            // Write device info
            this.writeDeviceInfo(pw, log.getDeviceType());
            
            // Write track resume
            this.writeTrackInfo(pw, track);

            pw.write("WAYPOINTS\n");
            // Write the header
            this.writeTrackHeader(pw);

            // Write the track Points
            segments=track.getSegments();
            segmentIterator=segments.iterator();            
            while (segmentIterator.hasNext())
            {
                segment=segmentIterator.next();
                points=segment.getWayPoints();
                
                pointIterator=points.iterator();
                
                while (pointIterator.hasNext())
                {
                    point=pointIterator.next();

                    this.writeTrackPoint(pw, point);
                }
            }            
            pw.write("\n");
            
            pw.write("TRACKPOINTS\n");
            // Write the header
            this.writeTrackHeader(pw);

            // Write the track Points
            segments=track.getSegments();
            segmentIterator=segments.iterator();            
            while (segmentIterator.hasNext())
            {
                segment=segmentIterator.next();
                points=segment.getTrackPoints();
                
                pointIterator=points.iterator();
                
                while (pointIterator.hasNext())
                {
                    point=pointIterator.next();

                    this.writeTrackPoint(pw, point);
                }
            }

            //Flush the output to the file
            pw.flush();

            //Close the Print Writer
            pw.close();

            //Close the File Writer
            fw.close();
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing to file "+fileName+": "+e.toString());
        }


    }

    
    /**
     * This method prints some info on the waypoint log to the csv file
     * @param writer Writer to write to
     * @param log Log to print info about
     */
    private void writeWaypointInfo(PrintWriter writer, WaypointLog log)
    {
        writer.print("Number of waypoints"+SEPARATOR+log.getNumberOfEntries()+"\n");
        writer.print("\n");
    }
    
    /**
     * This method prints the header for the waypoint log
     * @param writer Writer to write to
     */
    private void writeWaypointHeader(PrintWriter writer)
    {
        // Id
        writer.print("ID");
        writer.print(SEPARATOR);        
        
        // Latitude
        writer.print("Latitude (deg)");
        writer.print(SEPARATOR);        
        
        // Longitude
        writer.print("Longitude (deg)");
        writer.print(SEPARATOR);        
        
        // Elevation
        writer.print("Elevation (m)");
        
        writer.print("\n");        
    }
    
    /**
     * This method writes the waypoint to csv file
     * @param writer Writer to use
     * @param point Point to write
     */
    private void writeWaypoint(PrintWriter writer, Waypoint point)
    {
        // Id
        writer.print(point.getId());
        writer.print(SEPARATOR);        
        
        // Latitude
        writer.print(point.getLatitude());
        writer.print(SEPARATOR);        
        
        // Longitude
        writer.print(point.getLongitude());
        writer.print(SEPARATOR);        
        
        // Elevation
        writer.print(point.getElevation());
        
        writer.print("\n");
    }
    
    
    /**
     * This method writes the waypoints in the WaypointLog to file
     * @param fileName The filename
     */
    public void writeWaypointsToFile(String fileName)
    {
        WaypointLog             log;
        ArrayList<Waypoint>     points;
        Iterator<Waypoint>      iterator;
        Waypoint                point;
        
        log=WaypointLog.getInstance();
        points=log.getWaypoints();
        iterator=points.iterator();
        
        try
        {
            FileWriter fw = new FileWriter(fileName);
            PrintWriter pw = new PrintWriter(fw);

            // Write device info
            this.writeDeviceInfo(pw, log.getDeviceType());
            
            // Write log info
            this.writeWaypointInfo(pw, log);
            
            pw.write("WAYPOINT LOG\n");
            // Write the header
            this.writeWaypointHeader(pw);
                
            while (iterator.hasNext())
            {
                point=iterator.next();

                this.writeWaypoint(pw, point);
            }
   
            pw.write("\n");
            
            //Flush the output to the file
            pw.flush();

            //Close the Print Writer
            pw.close();

            //Close the File Writer
            fw.close();
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing to file "+fileName+": "+e.toString());
        }

        
    }
    
    
    /**
     * This method prints some info on the route log to the csv file
     * @param writer Writer to write to
     * @param log Log to print info about
     */
    private void writeRouteInfo(PrintWriter writer, RouteLog log)
    {
        writer.print("Number of route points"+SEPARATOR+log.getNumberOfEntries()+"\n");
        writer.print("\n");
    }    

    /**
     * This method writes the route point header
     * @param writer Writer to write to
     */
    private void writeRouteHeader(PrintWriter writer)
    {
        // Name
        writer.print("Name");
        writer.print(SEPARATOR);        
        
        // Latitude
        writer.print("Latitude (deg)");
        writer.print(SEPARATOR);        
        
        // Longitude
        writer.print("Longitude (deg)");
        writer.print(SEPARATOR);        
        
        // Elevation
        writer.print("Elevation (m)");
        writer.print(SEPARATOR);      
                
        // Symbol name
        writer.print("Symbol");
     
                
        writer.print("\n");        
    }
    
    /**
     * This method writes the route point as line in the csv file
     * @param writer Writer to write to
     * @param point Point to write
     */
    private void writeRoutePoint(PrintWriter writer, RoutePoint point)
    {
        // Name
        writer.print(point.getName());
        writer.print(SEPARATOR);        
        
          // Latitude
        writer.print(point.getLatitude());
        writer.print(SEPARATOR);        
        
        // Longitude
        writer.print(point.getLongitude());
        writer.print(SEPARATOR);        
        
        // Elevation
        writer.print(point.getElevation());
        writer.print(SEPARATOR);        
                
        // Symbol name
        writer.print(point.getSymbolName());
                
        writer.print("\n");
      
    }
    
    /**
     * The method writes the route in the RouteLog to file
     * @param fileName The name of the file to write to
     */
    public void writeRouteToFile(String fileName)
    {
        RouteLog                log;
        ArrayList<RoutePoint>   points;
        Iterator<RoutePoint>    iterator;
        RoutePoint              point;
        
        log=RouteLog.getInstance();
        points=log.getWaypoints();
        iterator=points.iterator();
        
        try
        {
            FileWriter fw = new FileWriter(fileName);
            PrintWriter pw = new PrintWriter(fw);

            // Write device info
            this.writeDeviceInfo(pw, log.getDeviceType());
            
            // Write some route log info
            this.writeRouteInfo(pw, log);
            
            pw.write("ROUTE\n");
            // Write the header
            this.writeRouteHeader(pw);
                
            while (iterator.hasNext())
            {
                point=iterator.next();

                this.writeRoutePoint(pw, point);
            }
   
            pw.write("\n");
 
            //Flush the output to the file
            pw.flush();

            //Close the Print Writer
            pw.close();

            //Close the File Writer
            fw.close();
        }
        catch(IOException e)
        {
            DebugLogger.error("Error writing Route to file "+fileName+": "+e.toString());
        }        
    }
    
    /**
     * The method writes the heartrate  of one track to file
     * @param fileName The name of the file to write to
     */
    public void writeHeartRateToFile(String fileName, int trackNo)
    {
        TrackLog                        log;
        List<HeartRatePoint>            rates;
        HeartRatePoint                  rate;
        ListIterator<HeartRatePoint>    iterator;
        
        
        log=TrackLog.getInstance();
        rates=log.getTrackHeartRateLog(trackNo);
        
        
        if (rates!=null)
        {
            try
            {
                FileWriter fw = new FileWriter(fileName);
                PrintWriter pw = new PrintWriter(fw);

                //Write to file for the first row
                pw.print("DateTime (UTC)");
                pw.print(",");
                pw.print("heart rate (bpm)\n");

                iterator=rates.listIterator();
                
                while (iterator.hasNext())
                {
                    rate=iterator.next();
                    if (rate.getHeartRateValue()>0)
                    {    
                        //Write to file for the second row
                        pw.print(rate.getDateTime().format("YYYY-MM-DD hh:mm:ss"));
                        pw.print(",");
                        pw.print(rate.getHeartRateValue());
                        pw.print("\n");
                    }

                }
                
                //Flush the output to the file
                pw.flush();

                //Close the Print Writer
                pw.close();

                //Close the File Writer
                fw.close();
            }
            catch(IOException e)
            {
                DebugLogger.error("Error writing to file "+fileName+": "+e.toString());
            }
        }
    }    
}
