/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package net.deepocean.u_gotme;

import java.io.*;
import java.util.ArrayList;


/**
 * This class implements the connection to a simulated i-gotU device. It inherits
 * from the Connection class which implements the protocol at application level.
 * This class reads the flash memory and command responses from files.
 * @author Jorgen
 */
public class ConnectionSimulation extends Connection
{
    private byte[] flashDump;
    private int    flashDumpLength;
    private byte[] responseAcknowledge;
    private int    responseAcknowledgeLength;
    private byte[] responseModel;
    private int    responseModelLength;
    private byte[] responseIdentification;
    private int    responseIdentificationLength;
    private byte[] responseCount;
    private int    responseCountLength;    
    
    boolean         isWriting;
    int             bytesToWrite;
    int             addressToWrite;    

    byte[]          response;
    int             responseOffset;
    int             responseLength;    
    int             count;
    
    /**
     * Constructor
     * @param simulationDataPath Path where to find the simulation files. 
     */
    public ConnectionSimulation(String simulationDataPath)
    {
        super();
        
       
        initialiseSimulationData(simulationDataPath);
        
        isWriting=false;
        addressToWrite=0;
        bytesToWrite=0;
    }
    
    /**
    *  This method returns a list of comport names
    */
    @Override
    public ArrayList<String> getComportList()
    {
        ArrayList<String>   ports;
        
        // Else, scan the comports
        ports=new ArrayList<String>();
        ports.add("simulation");
        return ports;
    }

    
    /**
     * This method reads data from a file to a buffer
     * @param fileName Name of the file to read
     * @param buffer Byte buffer to read the data into
     * @return Number of bytes read.
     */
    private int readData(String fileName, byte[] buffer)
    {
        int             bytesRead;
        FileInputStream inputStream;
        int             i;
        
        bytesRead=0;
        try 
        {
            inputStream = new FileInputStream(fileName);

            // read fills buffer with data and returns
            // the number of bytes read (which of course
            // may be less than the buffer size, but
            // it will never be more).
            bytesRead=inputStream.read(buffer);

            // Always close files.
            inputStream.close();	
            DebugLogger.info("Simulation: Read "+bytesRead+" bytes from "+fileName);

        }
        catch(FileNotFoundException ex) 
        {
            DebugLogger.error("Error opening simulation data file "+fileName);
        }
        catch(IOException ex) 
        {
            DebugLogger.error("Error reading simulation data file "+fileName);
        }
        return bytesRead;
    }
    
    /**
     * This method intializes the simulated device. It reads the flash memory
     * dump from file and the corresponding command responses.
     * @param simulationDataPath Path to look for the files.
     */
    private void initialiseSimulationData(String simulationDataPath)
    {
        String      fileName;

        flashDump=new byte[8388610];
        fileName=simulationDataPath+"\\memory_dump.dat";
        flashDumpLength=readData(fileName, flashDump);

        responseModel=new byte[100];
        fileName=simulationDataPath+"\\modelcmd_response.dat";
        responseModelLength=readData(fileName, responseModel);
        
        responseIdentification=new byte[100];
        fileName=simulationDataPath+"\\identificationcmd_response.dat";
        responseIdentificationLength=readData(fileName, responseIdentification);
        
        
        responseCount=new byte[100];
        fileName=simulationDataPath+"\\countcmd_response.dat";
        responseCountLength=readData(fileName, responseCount);
        
        responseAcknowledge=new byte[100];
        fileName=simulationDataPath+"\\acknowledge_response.dat";
        responseAcknowledgeLength=readData(fileName, responseAcknowledge);
        
    }
    

    /**
     * Helper function to make life easier. It compares the data array
     * against a human-more-readable string. The string consists of hex values
     * like "930504xxA0". Each two digits represent the hex value of a byte.
     * The value "xx" stands for any value.
     * @param data The data array
     * @param test The test string. Its length defines the length to compare
     * @return True if equal, false if not
     */
    private boolean isEqual(byte[] data, String test)
    {
        boolean equal;
        int i;
        int length;
        String subString;
        
        equal=true;
        
        length=test.length();
        
        i=0;
        while ((i<length) && equal)
        {
            subString=test.substring(i, i+2);
            if (!subString.equals("xx"))
            {
                if ((byte)Integer.parseInt(subString, 16)!=data[i/2])
                {
                    equal=false;
                }
            }
            i+=2;
        }
        
        
        return equal;
    }
    
    
    /**
     * This method handles the messages that are send during writing.
     * @param outputString Message
     * @param expectedDataLength Expected length of the message
     */
    private void handleWriteCommand(byte[] outputString, int expectedDataLength)
    {
        int i;
        int checkSum;
        
        i=0;
        checkSum=0;
        while (i<16)
        {
            checkSum+=(outputString[i]&0xff);
            i++;
        }
        if ((checkSum&0xff)==0 || true)
        {
            i=0;
            while ((i<15) && (bytesToWrite-i>0))
            {
                flashDump[addressToWrite+i]=outputString[i];
                count++;
                i++;
            }
            addressToWrite+=15;
            bytesToWrite-=Math.min(bytesToWrite, 15);

            response      =responseAcknowledge;
            responseLength=responseAcknowledgeLength-3;
            responseOffset=3;
            
            sendResponseHeader();
            sendResponseData();
            
        }
        else
        {
            response=null;
            DebugLogger.error("Simulation: Invalid write chunk: "+(checkSum&0xff));                   
        }        
        
   
        if (bytesToWrite<=0)
        {
            isWriting=false;            
        }
    }
    
    /**
     * Sends the data corresponding to the indicated response.
     */
    private void sendResponseData()
    {
        int i;
        
        // If the response was found, copy it to the public arrays
        if (response!=null)
        {
            
            i=0;
            while (i<responseLength)
            {
                responseData[i]=response[responseOffset+i];
                i++;
            }
            responseDataCount=responseLength;
        }          
    }
    
    /**
     *  Sends the header of the indicated response.
     */
    private void sendResponseHeader()
    {
        int i;
        
        // If the response was found, copy it to the public arrays
        if (response!=null)
        {
            
            i=0;
            while (i<3)
            {
                responseHeader[i]=response[i];
                i++;
            }
            responseHeaderCount=3;
        }         
    }
    
    
    /**
     * Function that writes the command to the device and waits for a response.
     * If waiting times out, retries are made.
     * After the action, following variables contain information
     *   responseHeaderCount -  Number of bytes in the header
     *   headerCount         -  Number of bytes in the response data
     *   responseHeader      -  Array containing the header (max. 3 bytes)
     *   responseData        -  Array containing the response data (max 4096 bytes)
     * @param outputString Byte array containing 16 bytes to send
     * @param expectedDataLength Expected data length of the response (excl. header)
     * @param crcRequired Indicates whether the last byte should be a crc
     * @return number of received bytes in responseData
     * @author Jorgen van der Velde
     */
    @Override
    public int outputAndWaitForResponse(byte[] outputString, int expectedDataLength)
    {

        int    i;

        int    checkSum;
        
        responseLength=0;
        responseOffset=0;
        response=null;

        if (isWriting)
        {
            handleWriteCommand(outputString, expectedDataLength);
        }
        else
        {
            // Model Command
            if (isEqual(outputString, "9305040003019F0000000000000000xx"))
            {
                response      =responseModel;
                responseOffset=3;
                responseLength=responseModelLength-3;
                sendResponseHeader();
                sendResponseData();
                DebugLogger.debug("Simulation: generating response to model command");
            }
            // Identification Command
            else if (isEqual(outputString, "930A00000000000000000000000000xx"))
            {
                response      =responseIdentification;
                responseOffset=3;
                responseLength=responseIdentificationLength-3;
                sendResponseHeader();
                sendResponseData();
                DebugLogger.debug("Simulation: generating response to identification command");
            }
            // Count Command
            else if (isEqual(outputString, "930B03001D00000000000000000000xx"))
            {
                response      =responseCount;
                responseOffset=3;
                responseLength=responseCountLength-3;
                sendResponseHeader();
                sendResponseData();
                DebugLogger.debug("Simulation: generating response to count command");
            }
            // NMEA switch Command
            else if (isEqual(outputString, "930101xx0000000000000000000000xx"))
            {
                response      =responseAcknowledge;
                responseOffset=3;
                responseLength=responseAcknowledgeLength-3;
                sendResponseHeader();
                sendResponseData();
                DebugLogger.debug("Simulation: generating response to NMEA switch command");
            }
            // read Command
            else if (isEqual(outputString, "930507xxxx0403xxxxxx0000000000xx"))
            {
                response      =flashDump;
                responseOffset=0;
                responseOffset|=(outputString[7]&0xff);
                responseOffset<<=8;
                responseOffset|=(outputString[8]&0xff);
                responseOffset<<=8;
                responseOffset|=(outputString[9]&0xff);
                responseLength=0;
                responseLength|=(outputString[3]&0xff);
                responseLength<<=8;
                responseLength|=(outputString[4]&0xff);

                // create response header
                responseHeader[0]=(byte)0x93;
                responseHeader[1]=outputString[3];
                responseHeader[2]=outputString[4];
                
                sendResponseData();
                
                DebugLogger.debug("Simulation: generating response to read command (0x"+
                                  String.format("%06x", responseOffset)+", 0x"+
                                  String.format("%04x", responseLength)+")");
            }
            // write command
            else if (isEqual(outputString, "930607xxxx0402xxxxxx0000000000xx"))
            {
                addressToWrite=0;
                addressToWrite|=(outputString[7]&0xff);
                addressToWrite<<=8;
                addressToWrite|=(outputString[8]&0xff);
                addressToWrite<<=8;
                addressToWrite|=(outputString[9]&0xff);
                bytesToWrite=0;
                bytesToWrite|=(outputString[3]&0xff);
                bytesToWrite<<=8;
                bytesToWrite|=(outputString[4]&0xff);

                count=0;
                isWriting=true;

                response      =responseAcknowledge;
                responseOffset=3;
                responseLength=responseAcknowledgeLength-3;

                sendResponseHeader();
                sendResponseData();
                
                DebugLogger.debug("Simulation: generating response to write command (0x"+
                                  String.format("%06x", addressToWrite)+", 0x"+
                                  String.format("%04x", bytesToWrite)+")");

            }
            // purge command (write commmand with writeCommand[6]=0x20)
            else if (isEqual(outputString, "93060700000420xxxxxx0000000000xx"))
            {
                addressToWrite=0;
                addressToWrite|=(outputString[7]&0xff);
                addressToWrite<<=8;
                addressToWrite|=(outputString[8]&0xff);
                addressToWrite<<=8;
                addressToWrite|=(outputString[9]&0xff);

                response      =responseAcknowledge;
                responseOffset=3;
                responseLength=responseAcknowledgeLength-3;

                sendResponseHeader();
                sendResponseData();
                
                DebugLogger.debug("Simulation: generating response to purge command (0x"+
                                  String.format("%06x", addressToWrite)+")");

            }
            // Unknown write command 1
            else if (isEqual(outputString, "93060400xx01060000000000000000xx"))
            {
                response      =responseAcknowledge;
                responseOffset=3;
                responseLength=responseAcknowledgeLength-3;
                sendResponseHeader();
                sendResponseData();
                DebugLogger.debug("Simulation: generating response to unknown write command 1");
            }            
            // Unknown write command 2
            else if (isEqual(outputString, "930504xxxx01050000000000000000xx"))
            {
                responseHeader[0]=(byte)0x93;
                responseHeader[1]=(byte)0x00;
                responseHeader[2]=(byte)0x01;
                responseHeaderCount=3;
                responseData[0]=0x00;
                responseDataCount=1;
                DebugLogger.debug("Simulation: generating response to unknown write command 2");
            }            
          
        }        
        
        return responseLength; 
    }        
}
