/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package net.deepocean.u_gotme;


import java.util.Enumeration;
import java.util.ArrayList;

/**
 * This class implements the connection to an i-gotU device via a comport.
 * It inherits from the Connection class, implementing the actual protocol at
 * application level.
 * @author Jorgen
 */
public abstract class ConnectionSerial extends Connection
{
    private enum                        ResponseType {UNKNOWN, ONEBLOCK, TWOBLOCK};
    
    /** Timeout for communication. Within this period response is expected
        after a request */
    private static final int            COMM_TIMEOUT_IN_MS      =2000;

    /** Maximum number of retries if no response */
    private static final int            MAX_RETRIES             =3;
    
    

    private Thread                      readThread;
    private String                      divertCode = "10";
    private static String               TimeStamp;
    
    
    
    
   
    /** Receive buffer */
    private byte[]                      buffer;

    
    // retry mechanism
    private boolean                     errorBlock=false;
    
    // Indicates that a command has been sent and a reponse is expected
    private boolean                     expectingResponse;
    private int                         localResponseCount;
    private int                         localResponseHeaderCount;
    private int                         localResponseDataCount;
    
    // Type of response: unknown, one block response or two block response
    private ResponseType                responseType;
    
    // Datalength as encoded in the response
    private int                         derivedDataLength;

    // Indicates timeout after receiving a response block
    private boolean                     timeout;

    // Response block counter
    private int                         blockCount;

    // Bytes not belonging to the response. It might be NMEA message
    private int                         crapBytesCount;
    
    public ConnectionSerial()
    {
        super();
   
        buffer                  =new byte[BLOCK_SIZE+HEADER_SIZE];

        expectedHeaderLength    =HEADER_SIZE;                 // header is always 3 bytes
        
        
        errorBlock              =false;
        expectingResponse       =false;
        timeout                 =false;
        blockCount              =0;
        responseType            =ResponseType.UNKNOWN;
    }
    
    /**
     * This method reads the bytes from the comport and writes them in the buffer
     * @param buffer Buffer to write bytes to
     * @return The number of bytes read
     */
    int comportRead(byte[] buffer)
    {
        return 0;
    }
    
    /**
     * This method writes bytes to the comport
     * @param buffer Bytes to write
     * @param length Number of bytes to write
     * @return False if succeeded, true if not
     */
    boolean comportWrite(byte[] buffer, int length)
    {
        return false;
    }
    
    
    protected void handeRxEvent()
    {
        synchronized(this)
        {
            int         bytesRead;
            int         i;
            boolean     endOfBlock;


            // We received some data. Now process it!
            bytesRead   =0;
            errorBlock  =false;
            endOfBlock  =false;


            // Read the bytes into the buffer
            bytesRead=comportRead(buffer);


            // Debug info
            if (bytesRead>0)
            {
                DebugLogger.debug("Device says 'Auch! Here are "+bytesRead+" bytes!'");
            }

/*                
            System.out.print("Bytes: ");
            i=0;
            while (i<bytesRead)
            {
                System.out.print(String.format("%02x ", buffer[i]));
                i++;
            }
            System.out.println();
*/                

            // Some devices GT-120 send two response blocks, the first always 
            // being 0x93 0x00 0x00. Second may contain the response data 0x93 xx xx yy yy ...
            // Other devices (GT-800+) send just
            // the second block and ommit the first.
            // Both blocks may be in the buffer. For large blocks
            // only part of the block (chunk) is in the buffer.

            // First check if we expect a response block. If not, we
            // might be receiving NMEA codes (GT-120)
            if (expectingResponse)
            {
                // Now process the buffer
                i=0;
                while ((i<bytesRead) && !errorBlock && blockCount<2)
                {
                    // The response header is 0x93 0xXX 0xXX
                    // The last two bytes are the data length 
                    if (localResponseHeaderCount==0)
                    {
                        // Check if the very first byte is 0x93.
                        // If not it might be some left over from a NMEA message
                        if (buffer[i]==(byte)0x93)
                        {
                            if (crapBytesCount>0)
                            {
                                DebugLogger.debug("Number of bytes preceding response: "+crapBytesCount);
                            }
                            responseHeader[localResponseHeaderCount]=buffer[i];
                            localResponseHeaderCount++;   
                            localResponseCount++;
                        }
                        else
                        {
                            crapBytesCount++;
                            // We might expect some rubish before the 1st 
                            // response block, but not after we've received a 
                            // valid response block.
                            if (blockCount>0)
                            {
                                DebugLogger.error("Unexpected data");
                                errorBlock=true;
                            }
                        }
                        endOfBlock=false;
                    }
                    else if (localResponseHeaderCount<3)
                    {
                        responseHeader[localResponseHeaderCount]=buffer[i];
                        endOfBlock=false;
                        localResponseHeaderCount++;
                        localResponseCount++;

                        // Additional check: datalength in header should match
                        // expected data length
                        if (localResponseHeaderCount==3)
                        {
                            // Get the length of the response data in the block
                            derivedDataLength=((responseHeader[1]&0xff)<<8) | (responseHeader[2]&0xff);


                            // It must equal the expected data length or 0 (GT-120, first response block)
                            if ((derivedDataLength!=expectedDataLength) && (derivedDataLength!=0))
                            {
                                DebugLogger.error("Inconsistent header");
                                errorBlock=true;
                            }

                            // If the derived data length is 0, this is it. No more data to be 
                            // expected in this response block
                            if (derivedDataLength==0)
                            {
                                DebugLogger.debug("Response block found, data length"+derivedDataLength);
                                responseCount           =localResponseCount;
                                responseHeaderCount     =localResponseHeaderCount;
                                responseDataCount       =localResponseDataCount;                                
                                localResponseCount      =0;
                                localResponseHeaderCount=0;
                                localResponseDataCount  =0;
                                endOfBlock              =true;
                                blockCount++;
                            }
                        }
                    }
                    else
                    {
                        // Data is response block data
/*                            
                        // If more data is received than expected...
                        // generate error. Also prevents buffer overflow.
                        // This sometimes happens in the Win version
                        // Aparently halfway the block an erronous event
                        // is generated...
                        if (localResponseDataCount>=expectedDataLength)
                        {
                            DebugLogger.error("More data received than expected!");
                            errorBlock=true;
                        }
                        else
                        {
                            responseData[localResponseDataCount]=buffer[i];
                            localResponseDataCount++;
                        }
*/
                        responseData[localResponseDataCount]=buffer[i];
                        localResponseDataCount++;
                        localResponseCount++;                            
                        endOfBlock=false;

                        // If the data received matches what we expect, we received
                        // a complete response block
                        if (localResponseDataCount==derivedDataLength)
                        {
                            DebugLogger.debug("Response block found, data length"+derivedDataLength);
                            responseCount           =localResponseCount;
                            responseHeaderCount     =localResponseHeaderCount;
                            responseDataCount       =localResponseDataCount; 
                            localResponseCount      =0;
                            localResponseHeaderCount=0;
                            localResponseDataCount  =0;
                            endOfBlock              =true;
                            blockCount++;                                
                        }

                    }

                    i++;
                }


                // Notify if all data has been received (header+data)
                if (errorBlock)
                {
                    this.notify();
                }

                // Check if the last byte received was the end of a 
                // response block
                if (endOfBlock)
                {
                    // If we expect one response block: notify
                    if (responseType==ResponseType.ONEBLOCK)
                    {
                        if (blockCount==1)
                        {
                            timeout=false;
                            this.notify();
                        }
                        else
                        {
                            DebugLogger.error("Found more response blocks while expecting one");
                        }

                    }
                    else if (responseType==ResponseType.TWOBLOCK)
                    {
                        if (blockCount==2)
                        {
                            timeout=false;
                            this.notify();
                            DebugLogger.debug ("Two block response - wakeup");
                        }
                    }
                    else if (responseType==ResponseType.UNKNOWN)
                    {
                        if (blockCount==2)
                        {
                            responseType=ResponseType.TWOBLOCK;
                            timeout=false;
                            this.notify();
                            DebugLogger.debug ("Two block response identified - wakeup");
                        }
                        else // first response block
                        {
                            // We received one block, but the data length is
                            // not 0. This means: one block response
                            if (responseDataCount>0)
                            {
                                // We received
                                responseType=ResponseType.ONEBLOCK;
                                timeout=false;
                                this.notify();
                                DebugLogger.debug ("One block response identified - wakeup");
                            }
                            else
                            {
                                // One response block received with 0 data bytes, 
                                // while we don't know
                                // whether the device sends one or two. Let it timeout
                                // unless a second block is received.
                                timeout=true;
                                DebugLogger.debug ("One block response - timeout");
                            }
                        }
                    }
                }


            }
        }
        
    }
    
    
    


    /**
     * Function that writes the command to the device and waits for a response.
     * If waiting times out, retries are made.
     * After the action, following variables contain information
     *   responseHeaderCount -  Number of bytes in the header
     *   headerCount         -  Number of bytes in the response data
     *   responseHeader      -  Array containing the header (max. 3 bytes)
     *   responseData        -  Array containing the response data (max 4096 bytes)
     * @param outputString Byte array containing 16 bytes to send
     * @param expectedDataLength Expected data length of the response (excl. header)
     * @param crcRequired Indicates whether the last byte should be a crc
     * @return number of received bytes in responseData
     * @author Jorgen van der Velde
     */
    @Override
    public int outputAndWaitForResponse(byte[] outputString, int expectedDataLength)
    {
        int         i;
        int         sum;
        boolean     ready;
        int         retries;
        int         dataLength;

        dataLength=0;
        retries=0;
        ready=false;

        // Calculate the checksum byte
        i=0;
        sum=0;
        while (i<15)
        {
            sum+=outputString[i];
            i++;
        }
        outputString[15]=(byte)(0x100-(sum&0xff));        
        
        while (!ready && retries<MAX_RETRIES)
        {
            synchronized(this)
            {
                this.localResponseCount         =0;
                this.localResponseHeaderCount   =0;
                this.localResponseDataCount     =0;
                this.responseCount              =0;
                this.responseHeaderCount        =0;
                this.responseDataCount          =0;
                this.expectedDataLength         =expectedDataLength;
                this.expectingResponse          =true;
                this.timeout                    =false;
                this.blockCount                 =0;
                this.crapBytesCount             =0;
            }

            DebugLogger.debug("Kicking devices' ass. (Retry "+retries+")");

            comportWrite(outputString, 16);

            synchronized(this)
            {
                // Waiting for response....
                try
                {
                    this.wait(COMM_TIMEOUT_IN_MS);
                    dataLength=responseDataCount;
                }
                catch(InterruptedException e)
                {
                }

                if (timeout)
                {
                        if (responseType==ResponseType.UNKNOWN)
                        {
                            responseType=ResponseType.ONEBLOCK;
                            DebugLogger.debug ("One block response identified");

                            // Woken up: response received or timed out
                            if (!errorBlock && (responseHeaderCount==3) && (responseDataCount==expectedDataLength))
                            {
                                ready=true;
                            }
                            else
                            {
                                DebugLogger.error("Error reading device. Error: "+errorBlock+", Header: "+responseHeaderCount+
                                                   " Data: "+responseDataCount+". I am going to retry");

                                retries++;
                            }
                        }
                        else
                        {
                          DebugLogger.error("No response from device or response to short");
                          dataLength=responseDataCount;
                        }
                            
                }
                
                
                // Woken up: response received or timed out
                if (!errorBlock && (responseHeaderCount==3) && (responseDataCount==expectedDataLength))
                {
                    ready=true;
                }
                else
                {
                    DebugLogger.error("Error reading device. Error: "+errorBlock+", Header: "+responseHeaderCount+
                                       " Data: "+responseDataCount+". I am going to retry");

                    retries++;
                }

            }
        }
        if (!ready)
        {
            DebugLogger.error("Wtf! Failed to communicate to device.");
        }
        return dataLength;
    }

    
}
