/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package net.deepocean.u_gotme;

import java.util.ArrayList;

/**
 * This class represents the connection to a device at application level. 
 * It implements the communication protocol to the i-gotU devices. 
 * It does not implement the data connection. The data connection must be added
 * in subclasses, by overriding the open(), close() and outputAndWaitForResponse() 
 * methods.
 * @author Jorgen
 */
public class Connection 
{
    public static final int    BLOCK_SIZE              =0x1000;
    public static final int    HEADER_SIZE             =0x03;    
    
    /** Number of bytes in one block written to the device */
    private static final int    WRITEBLOCK_SIZE         =0x100;

    /** Size of the chunk each block is divided in */
    private static final int    WRITECHUNK_SIZE         =0x10;    
 
    
    private byte[] identificationCommand ={(byte)0x93, (byte)0x0A, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};
    private byte[] modelCommand          ={(byte)0x93, (byte)0x05, (byte)0x04, (byte)0x00,
                                           (byte)0x03, (byte)0x01, (byte)0x9F, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};
    private byte[] nmeaSwitchCommand     ={(byte)0x93, (byte)0x01, (byte)0x01, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};
    private byte[] countCommand          ={(byte)0x93, (byte)0x0B, (byte)0x03, (byte)0x00,
                                           (byte)0x1D, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};
    private byte[] readCommand           ={(byte)0x93, (byte)0x05, (byte)0x07, (byte)0x10,
                                           (byte)0x00, (byte)0x04, (byte)0x03, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};

    private byte[] writeCommand          ={(byte)0x93, (byte)0x06, (byte)0x07, (byte)0x00,
                                           (byte)0x00, (byte)0x04, (byte)0x20, (byte)0x10,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};

    private byte[] unknownWriteCommand1  ={(byte)0x93, (byte)0x06, (byte)0x04, (byte)0x00,
                                           (byte)0x00, (byte)0x01, (byte)0x06, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};

    private byte[] unknownWriteCommand2  ={(byte)0x93, (byte)0x05, (byte)0x04, (byte)0x00,
                                           (byte)0x01, (byte)0x01, (byte)0x05, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};

    private byte[] unknownPurgeCommand1  ={(byte)0x93, (byte)0x0C, (byte)0x00, (byte)0x1E,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};

    private byte[] unknownPurgeCommand2  ={(byte)0x93, (byte)0x08, (byte)0x02, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
                                           (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00};
    
    
    
    
    
    
    /** The response header, you know, the 0x93 0xXX 0xYY thing */
    protected byte[]                    responseHeader;

    /** The response data */
    protected byte[]                    responseData;

    /** Number of bytes received */
    protected int                       responseCount;

    /** Number of header bytes receveived */
    protected int                       responseHeaderCount;

    /** Number of response data bytes received */
    protected int                       responseDataCount;

    
    
    /** Expected data to be received (excl. header) */
    protected int                       expectedDataLength;

    /** Expected header length (always 3) */
    protected int                       expectedHeaderLength;

    /**
     *  The constructor. Creates the databuffers.
     */
    public Connection()
    {
        responseHeader          =new byte[HEADER_SIZE];
        responseData            =new byte[BLOCK_SIZE];
    }
    
    /**
    *  This method returns a list of comports
    */
    public ArrayList<String> getComportList()
    {
        return null;
    }
    
    
   /**
     * Opens connection to the device
     * @param comportNumber Number of the comport to which the igotu is attached
     * @author Jorgen van der Velde
     */
 
    public void open(String comport)
    {
        
    }
    
    /**
     * This method closes the comport 
     */
    public void close()
    {
        
    }
    
    
        /**
     * Function that writes the command to the device and waits for a response.
     * If waiting times out, retries are made.
     * After the action, following variables contain information
     *   responseHeaderCount -  Number of bytes in the header
     *   headerCount         -  Number of bytes in the response data
     *   responseHeader      -  Array containing the header (max. 3 bytes)
     *   responseData        -  Array containing the response data (max 4096 bytes)
     * @param outputString Byte array containing 16 bytes to send
     * @param expectedDataLength Expected data length of the response (excl. header)
     * @param crcRequired Indicates whether the last byte should be a crc
     * @return number of received bytes in responseData
     * @author Jorgen van der Velde
     */
    public int outputAndWaitForResponse(byte[] outputString, int expectedDataLength)
    {
        return 0; 
    }    
    
    /**
     * This method returns the pointer to the response data array
     * @return Pointer to the response data array
     */
    byte[] getResponseData()
    {
        return this.responseData;
    }

    /**
     * This method returns the pointer to the response header array
     * @return Pointer to the response header array
     */
    byte[] getResponseHeader()
    {
        return this.responseHeader;
    }
    
    public int getResponseHeaderLength()
    {
        return this.responseHeaderCount;
    }
    
    public int getResponseDataLength()
    {
        return this.responseDataCount;
    }


    /**
     * This method executes the device identification command
     * @return The response length
     */
    public int commandGetIdentification()
    {
        return outputAndWaitForResponse(identificationCommand, 10);
    }

    /**
     * This method executes the device model command
     * @return Response length
     */
    public int commandGetModel()
    {
        return outputAndWaitForResponse(modelCommand, 3);
    }
    
    /**
     * This method executes the NMEA command
     * @param mode Mode to switch to
     * @return Response length
     */
    public int commandNmeaSwitch(byte mode)
    {
        nmeaSwitchCommand[3]=mode;

        return outputAndWaitForResponse(nmeaSwitchCommand, 0);        
    }
    
    /**
     * This method executes the count command
     * @return Response length
     */
    public int commmandCount()
    {
        return outputAndWaitForResponse(countCommand, 3);
    }
    
    
    /**
     * This command reads the flash memory
     * @param readPosition Position to read
     * @param size Number of bytes to read. Normally 0x10 or 0x1000
     * @return The response data length
     */
    public int commandReadFlash(int readPosition, int size)
    {
        int dataLength;
        
        readCommand[7]=(byte)((readPosition>>16)&0xff);
        readCommand[8]=(byte)((readPosition>> 8)&0xff);
        readCommand[9]=(byte)((readPosition    )&0xff);

        readCommand[3]=(byte)((size>> 8)&0xff);
        readCommand[4]=(byte)((size    )&0xff);
        
        dataLength=outputAndWaitForResponse(readCommand, size); 
        
        return dataLength;
    }
    
    public boolean commandWriteFlash(int writePosition, int size, byte[] writeData)
    {
        boolean isError;
        int     bytesToSend;
        byte[]  byteBuffer;
        int     i;
        int     blocks;
        int     chunks;
        int     block;
        int     chunk;
        int     address;
        int     blockByteCount;
        int     chunkByteCount;
        int     dataLength;



        isError=false;


        if (size%WRITEBLOCK_SIZE!=0)
        {
            DebugLogger.error("Size of data to write not a multiple of "+WRITEBLOCK_SIZE);
            isError=true;
        }

        if (!isError)
        {
            // Allocate some space for the chunks
            byteBuffer=new byte[WRITECHUNK_SIZE];

            // Number of write blocks
            blocks=size/WRITEBLOCK_SIZE;

            // configure write command
            // Block size. Should always be 0x10 (?)
            writeCommand[3]=(byte)((WRITEBLOCK_SIZE>>8)&0xFF);
            writeCommand[4]=(byte)((WRITEBLOCK_SIZE   )&0xFF);

            // Write command, always 0x02
            writeCommand[6]=(byte)0x02;


            // There we go: writing the blocks
            block=0;
            while (block<blocks && !isError)
            {
                address=writePosition+block*WRITEBLOCK_SIZE;
                writeCommand[7]=(byte)((address>>16)&0xFF);
                writeCommand[8]=(byte)((address>> 8)&0xFF);
                writeCommand[9]=(byte)((address    )&0xFF);

                DebugLogger.info("Send block to address 0x"+Integer.toHexString(address));
                // execute write command. We expect just the header
                dataLength=outputAndWaitForResponse(writeCommand, 0);
                if (dataLength!=0)
                {
                    isError=true;
                }
                
                
                if (!isError)
                {
                    blockByteCount=0;
                    chunkByteCount=0;
                    while (blockByteCount<WRITEBLOCK_SIZE)
                    {
                        // fill the chunk
                        byteBuffer[chunkByteCount]=writeData[block*WRITEBLOCK_SIZE+blockByteCount];

                        chunkByteCount++;

                        // chunk full? Write it
                        if (chunkByteCount==WRITECHUNK_SIZE-1)
                        {
                            dataLength=outputAndWaitForResponse(byteBuffer, 0);
                            chunkByteCount=0;
                        }
                        blockByteCount++;
                    }

                    // Write the last chunk if any data left in it
                    if (chunkByteCount>0)
                    {
                        // Fill rest of chunk with 0x00
                        while (chunkByteCount<WRITECHUNK_SIZE-1)
                        {
                            byteBuffer[chunkByteCount]=(byte)0x00;
                            chunkByteCount++;
                        }

                        dataLength=outputAndWaitForResponse(byteBuffer, 0);
                    }

                }
                if (isError)
                {
                    DebugLogger.error("Error writing block "+Integer.toHexString(writePosition));
                }
                block++;
            }
        }
        return isError; 
    }

    /**
     * Erases a 0x1000 block of flash memory
     * After action isError contains whether the action succeeded or not.
     * @param erasePosition Position in memory where to start erasing
     */
   public boolean commandEraseFlash(int erasePosition) 
   {

        int         bytesRead;
        boolean     exit;
        int         count;
        int         dataLength;
        boolean     isError;



        isError=false;

        // UNKNOWNWRITECOMMAND1
        // XX=0x00 for purge
        unknownWriteCommand1[4]=0x00;
        dataLength=outputAndWaitForResponse(unknownWriteCommand1, 0x00);
        if (dataLength!=0)
        {
            DebugLogger.error("Error on unknownWriteCommand1 while erasing block "+
                              Integer.toHexString(erasePosition));
            isError=true;
        }

        // WRITECOMMAND
        if (!isError)
        {
            // XX=0x0000 for purge
            writeCommand[3]=0x00;
            writeCommand[4]=0x00;

            // WW=0x20 for purge
            writeCommand[6]=0x20;

            // yy=memory address
            writeCommand[7]=(byte)((erasePosition>>16)&0xff);
            writeCommand[8]=(byte)((erasePosition>>8)&0xff);
            writeCommand[9]=(byte)((erasePosition)&0xff);

            dataLength=outputAndWaitForResponse(writeCommand, 0x00);
            if (dataLength!=0)
            {
                DebugLogger.error("Error on writeCommand while erasing block "+
                                  Integer.toHexString(erasePosition));

                isError=true;
            }
        }

        // UNKNOWNWRITECOMMAND2
        // Poll with UNKNOWNWRITECOMMAND2 until response is 0x00
        // In fact, it appears always to be 0x00 after 1st poll...
        exit=false;
        count=0;
        // XX=0x0001 for purge
        unknownWriteCommand2[3]=0x00;
        unknownWriteCommand2[4]=0x01;

        while (!exit && !isError)
        {
            dataLength=outputAndWaitForResponse(unknownWriteCommand2, 1);
            if (dataLength!=1)
            {
                DebugLogger.error("Error on unknownWriteCommand2 while erasing block "+
                                  Integer.toHexString(erasePosition));

                isError=true;
            }
            if (responseData[0]!=0x00 && !isError)
            {
                // Try at most 5 times x 200 ms = 1 sec
                // if response is not as expected
                if (count<5)
                {
                    try
                    {
                        Thread.sleep(200);
                    }
                    catch (InterruptedException e)
                    {
                        DebugLogger.error("Interrupted exception during Thread.sleep()");
                    }
                    count++;
                }
                else
                {
                    isError=true;
                    exit=true;
                }
            }
            else
            {
                exit=true;
            }
        }
        return isError;
   }
    

}
