/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package net.deepocean.u_gotme;

/**
 * This process is responsible for executing communication commands. It
 * runs a thread for that purpose.
 * @author Jorgen
 */
public class CommunicationProcess implements Runnable
{
    private static CommunicationProcess     theInstance=null;

    private boolean                         running;
    private boolean                         finished;
    private Command                         command;
    private ProgressListener                listener;
    private String                          parameter;
    private String                          outputString;

    private Thread                          theThread;
    private Device                          device;
    
    /**
     * The constructor. Creates the thread
     */
    private CommunicationProcess()
    {
        synchronized(this)
        {
            running=false;
            finished=false;
        }
        theThread=new Thread(this);
        theThread.start();

        device=Device.getInstance();
        
    }
    
    /**
     * Returns the one and only singleton instance of this class
     * @return The one and only instance of this class
     */
    public static CommunicationProcess getInstance()
    {
        if (theInstance==null)
        {
            theInstance=new CommunicationProcess();
        }
        return theInstance;
    }
    
    /**
     * This method executes a command. The data is stored and the thread is 
     * woken up.
     * @param command  The command to execute
     * @param listeren Progress listeren
     * @param param    Optional parameter to pass to the function to be executed.
     */
    void executeCommand(Command command, ProgressListener listener, String param)
    {
        synchronized(this)
        {
            this.listener       =listener;
            this.command        =command;
            this.parameter      =param;
            this.notify();
        }
    }

    /**
     * This method returns the result of the command
     * @return String describing the result.
     */
    String getResult()
    {
        String resultString;


        synchronized(this)
        {
            resultString=new String(outputString);
        }

        return resultString;
    }
    
    
    /**
     * Stop the process
     */
    void exit()
    {
        synchronized (this)
        {
            finished=true;
            this.notify();
        }
    }


    /**
     * Execute the command of downloading tracks
     * @param comport The comport to use
     * @param listener The listener to report progress to (0-99)
     */
    private void downloadTracks(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.downloadTracks(listener);
        device.close();
    }

    /**
     * Execute the command of downloading the list of waypoints
     * @param comport The comport to use
     * @param listener The listener to report progress to (0-99)
     */
    private void downloadWaypoints(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.downloadWaypoints(listener);
        device.close();
    }


    /**
     * Execute the command of downloading tracks
     * @param comport The comport to use
     * @param listener The listener to report progress to (0-99)
     */
    private void getInfo(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.getInfo(listener);
        device.close();
    }

    /**
     * Execute the command of erasing tracks
     * @param comport The comport to use
     * @param listener The listener to report progress to (0-99)
     */
    private void eraseTracks(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.eraseTracks(listener);
        device.close();
    }


    /**
     * This process establishes exactly the device type
     * @param comport The comport to use
     * @param listener The listener to report progress to (0-99)
     */
    private void getDeviceType(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.getDeviceType(listener);
        device.close();
    }

    /**
     * This method uploads a route
     * @param comport  Comport to use
     * @param listener Listener to use 
     */
    private void uploadRoute(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.uploadRoute(listener);
        device.close();
    }

    /**
     * This method downloads a route.
     * @param comport  Comport to use
     * @param listener Listener to use
     */
    private void downloadRoute(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.downloadRoute(listener);
        device.close();
    }
    
    /**
     * This method downloads a route.
     * @param comport  Comport to use
     * @param listener Listener to use
     */
    private void eraseRoute(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.eraseRoute(listener);
        device.close();
    }    

    /**
     * This method saves the simulation set. The parameter is passed as path 
     * where the files should be saved.
     * @param comport Comport to use
     * @param listener Listener to use
     */
    private void saveSimulationSet(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.simulationDump(parameter, listener);
        device.close();
    }
    
    /**
     * This method saves the Device settings to file. 
     * The parameter is passed as path where the file should be saved.
     * @param comport Comport to use
     * @param listener Listener to use
     */
    private void saveDeviceSettings(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.saveDeviceSettings(parameter, listener);
        device.close();
    }
    
    /**
     * This method restores the Device settings from file to the device. 
     * The parameter is passed as path where the file should be loaded from.
     * @param comport Comport to use
     * @param listener Listener to use
     */
    private void restoreDeviceSettings(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.restoreDeviceSettings(parameter, listener);
        device.close();
    }
    

    /**
     * This method saves the Device settings to file. 
     * The parameter is passed as path where the file should be saved.
     * @param comport Comport to use
     * @param listener Listener to use
     */
    private void saveDeviceSettingsAsText(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.saveDeviceSettingsAsText(parameter, listener);
        device.close();
    }
    
    /**
     * This method saves the Device settings to file. 
     * The parameter is passed as path where the file should be saved.
     * @param comport Comport to use
     * @param listener Listener to use
     */
    private void verifyCacheFile(String comport, ProgressListener listener)
    {
        device=Device.getInstance();
        device.open(comport);
        outputString=device.verifyCache(listener);
        device.close();
    }
    
    
    
    /**
     * The thread function. It goes to sleep and waits till a command needs
     * to be executed.
     */
    public void run()
    {
        boolean             finishedLocal;
        ProgressListener    listenerLocal;
        Command             commandLocal;

        DebugLogger.debug("Command executing thread started");
        synchronized(this)
        {
            running=true;
            finishedLocal=this.finished;
        }

        while (!finishedLocal)
        {
            synchronized(this)
            {
                try
                {
                    DebugLogger.debug("Command executing thread says 'Zzzzzzz'");
                    
                    this.wait();
                    DebugLogger.debug("Command executing thread says 'Wazzup?!'");
                }
                catch (InterruptedException e)
                {

                }

                // Get a local copy of the relevant variables
                // in a synchronized way
                finishedLocal=this.finished;
                listenerLocal=this.listener;
                commandLocal=new Command(command);

            }

            if (!finishedLocal)
            {
                switch (commandLocal.command)
                {
                    case COMMAND_GETINFO:
                        getInfo(commandLocal.comport, listenerLocal);
                        break;
                    case COMMAND_DOWNLOADTRACKS:
                        downloadTracks(commandLocal.comport, listenerLocal);
                        break;
                    case COMMAND_ERASETRACKS:
                        eraseTracks(commandLocal.comport, listenerLocal);
                        break;
                    case COMMAND_UPLOADROUTE:
                        uploadRoute(commandLocal.comport, listenerLocal);
                        break;
                    case COMMAND_DOWNLOADROUTE:
                        downloadRoute(commandLocal.comport, listenerLocal);
                        break;
                    case COMMAND_ERASEROUTE:
                        eraseRoute(commandLocal.comport, listenerLocal);
                        break;
                    case COMMAND_DOWNLOADWAYPOINTS:
                        downloadWaypoints(commandLocal.comport, listenerLocal);
                        break;
                    case COMMAND_GETDEVICETYPE:
                        this.getDeviceType(commandLocal.comport, listener);
                        break;
                    case COMMAND_SAVESIMULATIONSET:
                        this.saveSimulationSet(commandLocal.comport, listener);
                        break;
                    case COMMAND_SAVEDEVICESETTINGS:
                        this.saveDeviceSettings(commandLocal.comport, listener);
                        break;
                    case COMMAND_RESTOREDEVICESETTINGS:
                        this.restoreDeviceSettings(commandLocal.comport, listener);
                        break;
                    case COMMAND_SAVEDEVICESETTINGSASTEXT:
                        this.saveDeviceSettingsAsText(commandLocal.comport, listener);
                        break;
                    case COMMAND_VERIFYCACHEFILE:
                        this.verifyCacheFile(commandLocal.comport, listener);
                        break;
                }
                // signal the processing is finshed by writing 100%
                listenerLocal.reportProgress(100);
            }
        }
        DebugLogger.debug("Command executing thread says 'Basta! I quit'");

    }
}
